/////////////////////////////////////////////////////////////////////////////
//
//	File: QzUnitTest.cpp
//
//	$Header: /Projects/QzUnitTest/QzUnitTest.cpp  2  2009/9/14 1:50:32p  Lee $
//
//
//	Project Setting changes required to build:
//		change Character Set to Unicode
//		add "_WIN32_WINNT=0x0501" as a preprocessor definition
//		change runtime library to multithreaded
//
/////////////////////////////////////////////////////////////////////////////


#include <signal.h>
#include "QzCommon.h"
#include "QzDirList.h"
#include "QzLogger.h"
#include "QzMatrix4x4.h"
#include "UtfData.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


bool TestUtfTables(void);
bool TestUtfSize(void);
bool TestUtf32to08(void);
bool TestUtf32to16(void);
bool TestUtf16to08(void);
bool TestUtfNormalize(void);
bool TestUtfCompare(void);
bool TestUtfNumbers(void);
bool TestUtfCanonical(void);
bool TestUtfFormat(void);
bool TestUtfCasing(void);
bool TestUtfSorting(void);
bool TestUtfFile(void);
bool TestUtfHtmlConvert(void);
bool TestUtfWildcard(void);


/////////////////////////////////////////////////////////////////////////////
//
//	QzAssertHandler()
//
//	Custom assertion handler.  Deals with assertions without throwing up a
//	dialog, which doesn't necessary work right when running full-screen.
//
void QzAssertHandler(char message[], U32 lineNum, char file[])
{
	char *pattern = "Assert: line %1;, file %2;\n\n%3;\n";

	UtfFormat fmt;
	fmt.AddInt(lineNum);
	fmt.AddString(file);
	fmt.AddString(message);

	// Log the message.
	LogMessage(pattern, fmt);

	// Close the log file so we don't lose the assertion message.
	SafeDelete(g_pLog);

	// Use DebugBreak to launch the debugger ...
//	DebugBreak();

	// ... or just kill the app and drop back to the desktop.
	raise(SIGABRT);
}


/////////////////////////////////////////////////////////////////////////////
//
//	RandomFloat()
//
static float RandomFloat(void)
{
	return float(rand()) / float(RAND_MAX);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TestFloatConvert()
//
//	Verify that FloatToInt() correctly rounds off values, avoiding the
//	truncation problems that arise from simple float-to-int type casts.
//
//	Also verify that the various QzFloatIsZero...() routines properly
//	handle values around the threshold of their precision.
//
bool TestFloatConvert(void)
{
	U32 errorCount = 0;

	if (-3 != QzFloatToInt(-3.4f)) {
		++errorCount;
	}

	if (-3 != QzFloatToInt(-3.0f)) {
		++errorCount;
	}

	if (-3 != QzFloatToInt(-2.6f)) {
		++errorCount;
	}

	if (-1 != QzFloatToInt(-1.4f)) {
		++errorCount;
	}

	if (-1 != QzFloatToInt(-1.0f)) {
		++errorCount;
	}

	if (-1 != QzFloatToInt(-0.501f)) {
		++errorCount;
	}

	if (0 != QzFloatToInt(-0.499999f)) {
		++errorCount;
	}

	if (0 != QzFloatToInt(0.0f)) {
		++errorCount;
	}

	if (0 != QzFloatToInt(0.499999f)) {
		++errorCount;
	}

	if (7 != QzFloatToInt(6.501f)) {
		++errorCount;
	}

	if (7 != QzFloatToInt(7.0f)) {
		++errorCount;
	}

	if (7 != QzFloatToInt(7.49999f)) {
		++errorCount;
	}

	if (QzFloatIsZero10(0.500f / 1024.0f)) {
		++errorCount;
	}

	if (false == QzFloatIsZero10(0.499f / 1024.0f)) {
		++errorCount;
	}

	if (QzFloatIsZero16(0.500f / 65536.0f)) {
		++errorCount;
	}

	if (false == QzFloatIsZero16(0.499f / 65536.0f)) {
		++errorCount;
	}

	if (QzFloatIsZero20(0.500f / (1024.0f * 1024.0f))) {
		++errorCount;
	}

	if (false == QzFloatIsZero20(0.499f / (1024.0f * 1024.0f))) {
		++errorCount;
	}

	return (0 == errorCount);
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRandomMatrix()
//
void MakeRandomMatrix(QzMatrix4x4 &m)
{
	m.m_Matrix.c4x4[0][0] = RandomFloat();
	m.m_Matrix.c4x4[0][1] = RandomFloat();
	m.m_Matrix.c4x4[0][2] = RandomFloat();
	m.m_Matrix.c4x4[0][3] = 0.0f;
	m.m_Matrix.c4x4[1][0] = RandomFloat();
	m.m_Matrix.c4x4[1][1] = RandomFloat();
	m.m_Matrix.c4x4[1][2] = RandomFloat();
	m.m_Matrix.c4x4[1][3] = 0.0f;
	m.m_Matrix.c4x4[2][0] = RandomFloat();
	m.m_Matrix.c4x4[2][1] = RandomFloat();
	m.m_Matrix.c4x4[2][2] = RandomFloat();
	m.m_Matrix.c4x4[2][3] = 0.0f;
	m.m_Matrix.c4x4[3][0] = RandomFloat();
	m.m_Matrix.c4x4[3][1] = RandomFloat();
	m.m_Matrix.c4x4[3][2] = RandomFloat();
	m.m_Matrix.c4x4[3][3] = 1.0f;

}


/////////////////////////////////////////////////////////////////////////////
//
//	TestMatrixTransforms()
//
//	Verify that all of the simplified transforms yield the same results as
//	doing it the long way (generate a 4x4 matrix, then perform 4x4 matrix
//	multiply to compute the result).
//
bool TestMatrixTransforms(void)
{
	U32 errorCount = 0;

	UtfFormat fmt;

	srand(1087);

	QzMatrix4x4 base;
	QzMatrix4x4 rotX;
	QzMatrix4x4 rotY;
	QzMatrix4x4 rotZ;
	QzMatrix4x4 scale;
	QzMatrix4x4 trans;
	QzMatrix4x4 direct;
	QzMatrix4x4 product;

	for (U32 i = 0; i < 100; ++i) {
		MakeRandomMatrix(base);

		float angleX = RandomFloat() * 2.0f * c_PI;
		float angleY = RandomFloat() * 2.0f * c_PI;
		float angleZ = RandomFloat() * 2.0f * c_PI;

		float sx = 0.1f + (2.0f * RandomFloat());
		float sy = 0.1f + (2.0f * RandomFloat());
		float sz = 0.1f + (2.0f * RandomFloat());

		float tx = (2.0f * RandomFloat()) - 1.0f;
		float ty = (2.0f * RandomFloat()) - 1.0f;
		float tz = (2.0f * RandomFloat()) - 1.0f;

		rotX.MakeRotateX(angleX);
		rotY.MakeRotateY(angleY);
		rotZ.MakeRotateZ(angleZ);


		///////////////////
		//  PostRotateX  //
		///////////////////

		direct = base;
		direct.PostRotateX(angleX);

		product = base;
		product.PostMultiply(rotX);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PostRotateY  //
		///////////////////

		direct = base;
		direct.PostRotateY(angleY);

		product = base;
		product.PostMultiply(rotY);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PostRotateZ  //
		///////////////////

		direct = base;
		direct.PostRotateZ(angleZ);

		product = base;
		product.PostMultiply(rotZ);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PreRotateX  //
		///////////////////

		direct = base;
		direct.PreRotateX(angleX);

		product = base;
		product.PreMultiply(rotX);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PreRotateY  //
		///////////////////

		direct = base;
		direct.PreRotateY(angleY);

		product = base;
		product.PreMultiply(rotY);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PreRotateZ  //
		///////////////////

		direct = base;
		direct.PreRotateZ(angleZ);

		product = base;
		product.PreMultiply(rotZ);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		/////////////////
		//  PostScale  //
		/////////////////

		direct = base;
		direct.PostScale(sx);

		scale.MakeScale(sx);

		product = base;
		product.PostMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PostScale X  //
		///////////////////

		direct = base;
		direct.PostScale(sx, 1.0f, 1.0f);

		scale.MakeScale(sx, 1.0f, 1.0f);

		product = base;
		product.PostMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PostScale Y  //
		///////////////////

		direct = base;
		direct.PostScale(1.0f, sy, 1.0f);

		scale.MakeScale(1.0f, sy, 1.0f);

		product = base;
		product.PostMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		///////////////////
		//  PostScale Z  //
		///////////////////

		direct = base;
		direct.PostScale(1.0f, 1.0f, sz);

		scale.MakeScale(1.0f, 1.0f, sz);

		product = base;
		product.PostMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		////////////////
		//  PreScale  //
		////////////////

		direct = base;
		direct.PreScale(sx);

		scale.MakeScale(sx);

		product = base;
		product.PreMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		//////////////////
		//  PreScale X  //
		//////////////////

		direct = base;
		direct.PreScale(sx, 1.0f, 1.0f);

		scale.MakeScale(sx, 1.0f, 1.0f);

		product = base;
		product.PreMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		//////////////////
		//  PreScale Y  //
		//////////////////

		direct = base;
		direct.PreScale(1.0f, sy, 1.0f);

		scale.MakeScale(1.0f, sy, 1.0f);

		product = base;
		product.PreMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		//////////////////
		//  PreScale Z  //
		//////////////////

		direct = base;
		direct.PreScale(1.0f, 1.0f, sz);

		scale.MakeScale(1.0f, 1.0f, sz);

		product = base;
		product.PreMultiply(scale);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		/////////////////////
		//  PostTranslate  //
		/////////////////////

		direct = base;
		direct.PostTranslate(tx, ty, tz);

		trans.MakeTranslate(tx, ty, tz);

		product = base;
		product.PostMultiply(trans);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}


		////////////////////
		//  PreTranslate  //
		////////////////////

		direct = base;
		direct.PreTranslate(tx, ty, tz);

		trans.MakeTranslate(tx, ty, tz);

		product = base;
		product.PreMultiply(trans);

		if (false == direct.IsEqual(product)) {
			++errorCount;
		}
	}

	return (0 == errorCount);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TestMatrixPerformance()
//
//	Measure how long the different matrix transforms require.
//
bool TestMatrixPerformance(void)
{
	UtfFormat fmt;

	QzMatrix4x4 base;
	QzMatrix4x4 direct;
	QzMatrix4x4 product;

	srand(1087);

	MakeRandomMatrix(base);
	MakeRandomMatrix(direct);

	float tx = (2.0f * RandomFloat()) - 1.0f;
	float ty = (2.0f * RandomFloat()) - 1.0f;
	float tz = (2.0f * RandomFloat()) - 1.0f;

	float s1 = RandomFloat();
	float s2 = RandomFloat();
	float c1 = RandomFloat();
	float c2 = RandomFloat();

	QzVector axis;
	axis.m_X = (2.0f * RandomFloat()) - 1.0f;
	axis.m_Y = (2.0f * RandomFloat()) - 1.0f;
	axis.m_Z = (2.0f * RandomFloat()) - 1.0f;
	axis.Normalize();

	U64 t0, t1;

	const U32   loopSize   = 100000;
	const float makeMicro  = float(1000000.0 / (double(QzPrecisionClockFrequency()) * 8.0 * double(loopSize)));


	/////////////////////
	//  Sine + Cosine  //
	/////////////////////

	t0 = QzPrecisionClockRead();

	// Average time to perform one sin() and one cos().
	//
	// This is 8 calls to sin() and 8 to cos(), since we want to know the
	// total time between them: this is critical, since most rotations
	// require one call to sin() and one to cos(), which account for the
	// majority of CPU time needed for the transform.
	//
	// Note that this code must take the output of each function call and
	// feed it back in a few calls later, otherwise the compiler will
	// optimize the calls out.  And alternating the variables allows the
	// operations to be pipelined better -- arguably, this prevents us from
	// getting an accurate measurement of the individual calls, but we're
	// measuring sinf+cosf in the context of how they perform versus the
	// rotation functions.  In the rotation functions, there is no order
	// dependence, so the compiler will reorder the instructions for better
	// performance, so we want the compiler to do the same thing here to
	// better measure just how much of those operations depends on the sin+cos
	// and what is required for everything else that happens in those functions.
	//
	for (U32 i = 0; i < loopSize; ++i) {
		s1 = sinf(s1);
		c1 = cosf(c1);
		s2 = sinf(s2);
		c2 = cosf(c2);
		s1 = sinf(s1);
		c1 = cosf(c1);
		s2 = sinf(s2);
		c2 = cosf(c2);
		s1 = sinf(s1);
		c1 = cosf(c1);
		s2 = sinf(s2);
		c2 = cosf(c2);
		s1 = sinf(s1);
		c1 = cosf(c1);
		s2 = sinf(s2);
		c2 = cosf(c2);
	}

	// Trick the compiler into thinking we're doing something with the
	// results, otherwise the it could still optimize out the contents
	// of the loop, preventing us from getting a measurement.
	//
	product.MakeTranslate(s1, c1, 0.0f);
	product.MakeTranslate(s2, c2, 0.0f);

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("sin+cos       = %1; microseconds", fmt);


	////////////////////
	//  MakeIdentity  //
	////////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeIdentity();
		product.MakeIdentity();
		product.MakeIdentity();
		product.MakeIdentity();
		product.MakeIdentity();
		product.MakeIdentity();
		product.MakeIdentity();
		product.MakeIdentity();
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeIdentity  = %1; microseconds", fmt);


	///////////////////
	//  MakeRotateX  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeRotateX(tx * 2 * c_PI);
		product.MakeRotateX(ty * 2 * c_PI);
		product.MakeRotateX(tz * 2 * c_PI);
		product.MakeRotateX(tx * 2 * c_PI);
		product.MakeRotateX(ty * 2 * c_PI);
		product.MakeRotateX(tz * 2 * c_PI);
		product.MakeRotateX(tx * 2 * c_PI);
		product.MakeRotateX(ty * 2 * c_PI);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeRotateX   = %1; microseconds", fmt);


	///////////////////
	//  MakeRotateY  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeRotateY(tx * 2 * c_PI);
		product.MakeRotateY(ty * 2 * c_PI);
		product.MakeRotateY(tz * 2 * c_PI);
		product.MakeRotateY(tx * 2 * c_PI);
		product.MakeRotateY(ty * 2 * c_PI);
		product.MakeRotateY(tz * 2 * c_PI);
		product.MakeRotateY(tx * 2 * c_PI);
		product.MakeRotateY(ty * 2 * c_PI);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeRotateY   = %1; microseconds", fmt);


	///////////////////
	//  MakeRotateZ  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeRotateZ(tx * 2 * c_PI);
		product.MakeRotateZ(ty * 2 * c_PI);
		product.MakeRotateZ(tz * 2 * c_PI);
		product.MakeRotateZ(tx * 2 * c_PI);
		product.MakeRotateZ(ty * 2 * c_PI);
		product.MakeRotateZ(tz * 2 * c_PI);
		product.MakeRotateZ(tx * 2 * c_PI);
		product.MakeRotateZ(ty * 2 * c_PI);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeRotateZ   = %1; microseconds", fmt);


	///////////////////////////////
	//  MakeRotateArbitraryAxis  //
	///////////////////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeRotateArbitraryAxis(axis, tx * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, ty * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, tz * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, tx * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, ty * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, tz * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, tx * 2 * c_PI);
		product.MakeRotateArbitraryAxis(axis, ty * 2 * c_PI);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeArbAxis   = %1; microseconds", fmt);


	/////////////////
	//  MakeScale  //
	/////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
		product.MakeScale(tx, ty, tz);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeScale     = %1; microseconds", fmt);


	/////////////////////
	//  MakeTranslate  //
	/////////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
		product.MakeTranslate(tx, ty, tz);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("MakeTranslate = %1; microseconds", fmt);


	////////////////
	//  Multiply  //
	////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product.Multiply(base, direct);
		product.Multiply(base, direct);
		product.Multiply(base, direct);
		product.Multiply(base, direct);
		product.Multiply(base, direct);
		product.Multiply(base, direct);
		product.Multiply(base, direct);
		product.Multiply(base, direct);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("Multiply      = %1; microseconds", fmt);


	////////////////////
	//  PostMultiply  //
	////////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product = base;
		product.PostMultiply(direct);
		product.PostMultiply(direct);
		product.PostMultiply(direct);
		product.PostMultiply(direct);
		product.PostMultiply(direct);
		product.PostMultiply(direct);
		product.PostMultiply(direct);
		product.PostMultiply(direct);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PostMultiply  = %1; microseconds", fmt);


	///////////////////
	//  PreMultiply  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		product = base;
		product.PreMultiply(direct);
		product.PreMultiply(direct);
		product.PreMultiply(direct);
		product.PreMultiply(direct);
		product.PreMultiply(direct);
		product.PreMultiply(direct);
		product.PreMultiply(direct);
		product.PreMultiply(direct);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PreMultiply   = %1; microseconds", fmt);


	///////////////////
	//  PostRotateX  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
		direct.PostRotateX(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PostRotateX   = %1; microseconds", fmt);


	///////////////////
	//  PreRotateX  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
		direct.PreRotateX(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PreRotateX    = %1; microseconds", fmt);


	///////////////////
	//  PostRotateY  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
		direct.PostRotateY(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PostRotateY   = %1; microseconds", fmt);


	///////////////////
	//  PreRotateY  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
		direct.PreRotateY(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PreRotateY    = %1; microseconds", fmt);


	///////////////////
	//  PostRotateZ  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
		direct.PostRotateZ(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PostRotateZ   = %1; microseconds", fmt);


	///////////////////
	//  PreRotateZ  //
	///////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
		direct.PreRotateZ(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PreRotateZ    = %1; microseconds", fmt);


	/////////////////
	//  PostScale  //
	/////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PostScale(tx);
		direct.PostScale(tx);
		direct.PostScale(tx);
		direct.PostScale(tx);
		direct.PostScale(tx);
		direct.PostScale(tx);
		direct.PostScale(tx);
		direct.PostScale(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PostScale     = %1; microseconds", fmt);


	////////////////
	//  PreScale  //
	////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PreScale(tx);
		direct.PreScale(tx);
		direct.PreScale(tx);
		direct.PreScale(tx);
		direct.PreScale(tx);
		direct.PreScale(tx);
		direct.PreScale(tx);
		direct.PreScale(tx);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PreScale      = %1; microseconds", fmt);


	/////////////////////
	//  PostTranslate  //
	/////////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
		direct.PostTranslate(tx, ty, tz);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PostTranslate = %1; microseconds", fmt);


	////////////////////
	//  PreTranslate  //
	////////////////////

	t0 = QzPrecisionClockRead();

	for (U32 i = 0; i < loopSize; ++i) {
		direct = base;
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
		direct.PreTranslate(tx, ty, tz);
	}

	t1 = QzPrecisionClockRead();

	fmt.Reset();
	fmt.AddFloat(float(t1 - t0) * makeMicro);
	LogMessage("PreTranslate  = %1; microseconds", fmt);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	TestMatrix4x4Inverse()
//
//	Test matrix inversion.  When inverted, does the result yield an identity
//	matrix when multiplied by the original?  Can we safely extract normal
//	matrices from arbitrary transform matrices?
//
//	This test suggests that both answers are "Yes", especially when dealing
//	with matrices produced only from rotations, translations, and scalings.
//
bool TestMatrix4x4Inverse(void)
{
	U32 errorCount = 0;

	srand(10437);

	for (U32 i = 0; i < 30000; ++i) {
	//	QzVector axis(0.0f, 1.0f, 0.0f);
	//	QzVector axis(3.7f, 8.1f, 5.4f);
		QzVector axis(RandomFloat() + 0.01f, RandomFloat(), RandomFloat());
		axis.Normalize();

		QzQuat quat;
	//	quat.MakeFromAxisAngle(axis, 0.6143f);
	//	quat.MakeFromAxisAngle(axis, c_PI / 2.0f);
		quat.MakeFromAxisAngle(axis, RandomFloat() * c_PI);

		QzMatrix4x4 m1;
	//	m1.MakeTranslate(QzVector(1.3f, 2.9f, 7.1f));
		m1.MakeTranslate(QzVector(RandomFloat(), RandomFloat(), RandomFloat()));
		m1.PostRotate(quat);
		m1.PostScale(RandomFloat() + 0.5f, RandomFloat() + 0.5f, RandomFloat() + 0.5f);
	//	m1.PostRotateY(c_PI / 2.0f);

//		printf("base matrix:\n");
//		m1.PrintMatrix();

		QzMatrix4x4 inv;
		inv.ComputeInverseMatrix(m1);
//		printf("inverse matrix:\n");
//		inv.PrintMatrix();

		QzMatrix4x4 ident;
		ident.Multiply(m1, inv);

		if (false == ident.IsIdentity()) {
			++errorCount;
		}

//		printf("base * inverse:\n");
//		ident.PrintMatrix();

//		printf("identity: %s\n\n", ident.IsIdentity() ? "true" : "FALSE");

	//	QzVector vecsrc(1.0f, 0.0f, 0.0f);
	//	QzVector vecsrc(3.2f, 7.6f, -4.9f);
		QzVector vecsrc(RandomFloat() + 0.01f, RandomFloat(), RandomFloat());
		vecsrc.Normalize();

		QzVector vecnorm;

		QzMatrix4x4 norm;
		norm.ComputeNormalMatrix(m1);
//		printf("normal matrix:\n");
//		norm.PrintMatrix();
		norm.TransformNormal(vecsrc, vecnorm);


//		printf("vector: %8.5f %8.5f %8.5f = %f\n", vecsrc.m_X, vecsrc.m_Y, vecsrc.m_Z, vecsrc.Length());
//		printf("vector: %8.5f %8.5f %8.5f = %f\n", vecnorm.m_X, vecnorm.m_Y, vecnorm.m_Z, vecnorm.Length());

		// Empirically, this fails on 18 out of 30,000 attempts when testing
		// with QzEqualFloats16().  The bad exponents are between 110 and 114,
		// that badly degenerate matrices don't exist, and that the worst
		// error is still pretty darned small (0.0002).  Even trying with
		// QzEqualFloats20() only yields 22 out of 30,000 mismatches.
		//
		// In other words, the matrix generated by ComputeNormalMatrix should
		// always safely transform a normal vector and preserve its unit
		// length.  The transformed normal does not need to be renormalized
		// after the transform.  This may not hold true when given a purely
		// random matrix, but a matrix composed of only rotations, scalings,
		// and translations should not be degenerate.
		//
		if (false == QzEqualFloats10(1.0f, vecnorm.Length())) {
			++errorCount;
			float f = 1.0f - vecnorm.Length();
			printf("%d ", (*((U32*)&f) >> 23) & 0xFF);
			printf("%s %s %f\n", ident.IsIdentity() ? "true " : "FALSE",
					QzEqualFloats20(1.0f, vecnorm.Length()) ? "true " : "FALSE",
					vecnorm.Length());
		}
/*
		printf("%s %s %f\n", ident.IsIdentity() ? "true " : "FALSE",
				QzEqualFloats20(1.0f, vecnorm.Length()) ? "true " : "FALSE",
				vecnorm.Length());
*/
	}

	return (0 == errorCount);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TestTimePrecision()
//
//	How accurate and precise are the low- and high-precision clocks in the
//	system?
//
//	For Windows, when timeBeginPeriod is set to 5 milliseconds, timeGetTime
//	will consistently step by 3 or 4 milliseconds on every system that was
//	tested.  Granted, this was with nothing else running on the system, so
//	the results may be less consistent when there is a heavy load on the CPU.
//
//	QueryPerformanceCounter on the other hand, tends to vary widely in value
//	between calls.  Granted, some of that variation may be due to other
//	processes taking over the CPU, but this was true even on multi-core and
//	hyperthreaded systems.  Could rewrite this test to use bins, tracking
//	the different delta values that are computed to get an idea of the
//	distribution of samples.
//
//	For the Macbook, this indicates that Microsecond advances every single
//	microsecond, without ever skipping a value.  But only when the app has
//	the CPU -- when context switched out, the advance of the clock will be
//	missed and the clock will advance several milliseconds by the time this
//	process returns.  However, these tests are short enough in duration a
//	context switch happens infrequently.
//
bool TestTimePrecision(void)
{
	U32 baseTick    = QzGetMilliseconds();
	U32 prevTick    = baseTick;
	U32 updateCount = 0;
	U32 minDelta    = 0xFFFFFFFF;
	U32 maxDelta    = 0;

	while (updateCount < 100) {
		U32 newTick = QzGetMilliseconds();

		if (newTick != prevTick) {
			U32 delta = newTick - prevTick;
			if (minDelta > delta) {
				minDelta = delta;
			}
			if (maxDelta < delta) {
				maxDelta = delta;
			}
			prevTick = newTick;
			++updateCount;
		}
	}

	UtfFormat fmt;
	fmt.AddInt(prevTick - baseTick);
	fmt.AddFloat(float(prevTick - baseTick) / float(updateCount));
	fmt.AddInt(minDelta);
	fmt.AddInt(maxDelta);
	LogMessage("QzGetMillisecond() precision: range %1; avg = %2w1.3; ms, delta %3; to %4;", fmt);

	U64 baseTick64    = QzPrecisionClockRead();
	U64 prevTick64    = baseTick64;
	U64 minDelta64    = 0xFFFFFFFF;
	U64 maxDelta64    = 0;

	updateCount = 0;

	while (updateCount < 100) {
		U64 newTick = QzPrecisionClockRead();

		if (newTick != prevTick64) {
			U64 delta = newTick - prevTick64;
			if (minDelta64 > delta) {
				minDelta64 = delta;
			}
			if (maxDelta64 < delta) {
				maxDelta64 = delta;
			}
			prevTick64 = newTick;
			++updateCount;
		}
	}

	U64 frequency = QzPrecisionClockFrequency();

	fmt.Reset();
	fmt.AddInt(U32(prevTick64 - baseTick64));
	fmt.AddFloat((float(prevTick64 - baseTick64) * 1000.0f) / (float(frequency) * float(updateCount)));
	fmt.AddInt(U32(minDelta64));
	fmt.AddInt(U32(maxDelta64));
	LogMessage("QzPrecisionClockRead() precision: range %1;, avg = %2w1.3; ms, delta %3; to %4;", fmt);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	TestVector()
//
bool TestVector(void)
{
	U32 errorCount = 0;

	srand(31941);

	// Question: If a vector is normalized, how close is the length to 1.0?
	//
	// This is useful to know when normalizing.  If the length-squared of
	// the vector is within a threshold of 1.0, we can assume it is already
	// normalized and that we don't need to spend the expensive square root
	// operation.  After all, if we normalize a vector that is already close
	// to unit length, the change in magnitude will be negligible, and maybe
	// irrelevant due to the precision of floats.

	U32 expMin = 255;
	U32 expMax =   0;

	for (U32 i = 0; i < 30000; ++i) {
		float x = (RandomFloat() - 0.5f) * 1000.0f;
		float y = (RandomFloat() - 0.5f) *  100.0f;
		float z = (RandomFloat() - 0.5f) *   10.0f;
		QzVector v(x, y, z);
		v.Normalize();
		float ls = v.LengthSquared();
		float f  = ls - 1.0f;
		U32 exponent = ((*reinterpret_cast<U32*>(&f)) >> 23) & 0xFF;

		if (expMin > exponent) {
			expMin = exponent;
		}

		if (expMax < exponent) {
			expMax = exponent;
		}

		// Experimental results: The range of the exponent is between 88 and
		// 105, inclusive.  The exact lower range varies a bit, depending
		// upon the relative magnitudes of the vector's components.
		//
		// But this range is good enough to indicate that QzEqualFloats20()
		// will always detect a normalized vector, so we don't need to
		// normalize again.
		//
		if (false == QzEqualFloats20(1.0f, ls)) {
			++errorCount;
		}
	}

	if (expMin > 106) {
		++errorCount;
	}
	if (expMax > 106) {
		++errorCount;
	}

	return (0 == errorCount);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TestDirList()
//
//	For demo purposes, iterate all CPP files in the directory.
//
//	This test is only here for the file iteration article:
//		http://www.teachsolaisgames.com/articles/crossplat3.html
//
bool TestDirList(void)
{
	QzDirList dir;

	// Before scanning the directory, set these flags to indicate what
	// types of properties should be reported.
	dir.m_ShowDirs      = true;
	dir.m_ShowFiles     = true;
	dir.m_ShowHidden    = false;
	dir.m_ShowReadOnly  = true;
	dir.m_ShowSystem    = false;

	// Now scan the directory and write the names out to the log file.
	dir.ScanDirectory(CharToUtf("."), NULL);

	UtfFormat fmt;

	do {
		fmt.Reset();
		fmt.AddString(dir.m_Path);
		LogMessage("path: %1;", fmt);
		for (U32 i = 0; i < dir.m_EntryCount; ++i) {
			Utf08_t datestr[128];
			QzTimeConvertCrtToString(dir.m_pList[i].Time, datestr, ArraySize(datestr));
			fmt.Reset();
			fmt.AddInt(i + 1);
			fmt.AddString(datestr);
			fmt.AddInt(dir.m_pList[i].Size);
			fmt.AddString(dir.m_pList[i].pName);
			LogMessage("file %1w3;: %2; %3w9;, %4;", fmt);
		}
	} while (dir.UpOneLevel());

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
typedef bool (*UnitTestFunc_t)(void);

struct UnitTestList_t
{
	UnitTestFunc_t	FuncPtr;
	char*			Name;
};

static UnitTestList_t g_TestFuncs[] =
{
	{ TestFloatConvert,			"TestFloatConvert" },
	{ TestMatrixTransforms,		"TestMatrixTransforms" },
	{ TestMatrixPerformance,	"TestMatrixPerformance" },
	{ TestMatrix4x4Inverse,		"TestMatrix4x4Inverse" },
	{ TestTimePrecision,		"TestTimePrecision" },
	{ TestVector,				"TestVector" },

	{ TestDirList,				"TestDirList" },

	{ TestUtfTables,			"TestUtfTables" },
	{ TestUtfSize,				"TestUtfSize" },
	{ TestUtf32to08,			"TestUtf32to08" },
	{ TestUtf32to16,			"TestUtf32to16" },
	{ TestUtf16to08,			"TestUtf16to08" },
	{ TestUtfNormalize,			"TestUtfNormalize" },
	{ TestUtfCompare,			"TestUtfCompare" },
	{ TestUtfNumbers,			"TestUtfNumbers" },
	{ TestUtfCanonical,			"TestUtfCanonical" },
	{ TestUtfFormat,			"TestUtfFormat" },
	{ TestUtfCasing,			"TestUtfCasing" },
	{ TestUtfSorting,			"TestUtfSorting" },
	{ TestUtfFile,				"TestUtfFile" },
	{ TestUtfHtmlConvert,		"TestUtfHtmlConvert" },
	{ TestUtfWildcard,			"TestUtfWildcard" },
};


/////////////////////////////////////////////////////////////////////////////
//
int main(int argc, char *argv[])
{
	QzSystemInit();
	UtfInitialize();

	QzLogger *pLog = new QzLogger("QzLog");
	g_pLog = pLog;

	// Only enable file logging if the memory logging option is not available.
	// This typically means the logger utility has not been started.
	if (!g_pLog->MemoryLoggingAvailable()) {
		g_pLog->Open(reinterpret_cast<const Utf08_t*>("trace.txt"));
	}

	// Make certain temp directory exists.
	QzMakeDirectory(reinterpret_cast<const Utf08_t*>("Scratch"));

	U32 passCount = 0;
	U32 failCount = 0;

	for (U32 funcNum = 0; funcNum < ArraySize(g_TestFuncs); ++funcNum) {
		if (g_TestFuncs[funcNum].FuncPtr()) {
			++passCount;
			printf(".");
		}
		else {
			++failCount;
			printf("Error: %s() failed test\n", g_TestFuncs[funcNum].Name);
		}
	}

	printf("\nPassed: %d\nFailed: %d\n", passCount, failCount);

	g_pLog = NULL;

	SafeDelete(pLog);

	UtfUninitialize();
	QzSystemUninit();

#ifdef _DEBUG
	if (_CrtDumpMemoryLeaks()) {
		printf("ERROR: MEMORY LEAKS DETECTED\n");
	}
#endif

	return 0;
}



