/////////////////////////////////////////////////////////////////////////////
//
//	File: TestThread.cpp
//
//	$Header: /Projects/QzTest/TestThread.cpp  2  2009/9/14 1:43:53p  Lee $
//
//
//	This is a minimalist example of creating a thread that periodically
//	wakes up to do something, and also responds promptly when requested to
//	terminate.
//
//	The only work being done is to cycle a counter through four values,
//	which are used to create an ASCII-style spinning character.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzInterface.h"
#include "TestThread.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
TestThread::TestThread(void)
	:	m_Counter(0)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
TestThread::~TestThread(void)
{
	// Always make certain that the worker thread has been terminated.
	DestroyWorkerThread();
}


/////////////////////////////////////////////////////////////////////////////
//
//	CreateWorkerThread()
//
//	Call this method to start the worker thread.
//
//	Do not create a new thread if one is already active.  Unless your code
//	is repeatedly creating and destroying threads (which can be very
//	inefficient, depending on how long threads stay alive), attempting to
//	create a new thread when one already exists is a bug.
//
void TestThread::CreateWorkerThread(void)
{
	// Spawn a new thread.  This requires a static function pointer that
	// will be called by the OS, a context pointer (*this), and a name for
	// the thread.  The name is useful when running on Windows, where the
	// thread name can be displayed in the debugger.  There is no way in
	// Mac OS (that I know of) to assign a name to a thread.
	m_Thread.StartThread(StaticThreadFunc, this, "ThreadFunc");
}


/////////////////////////////////////////////////////////////////////////////
//
//	DestroyWorkerThread()
//
//	This is called during cleanup and at destruction, so care needs to be
//	taken to not destroy a thread that has already been destroyed.
//
//	Normally you would want to kill all worker threads in an orderly manner
//	to assure that they are finished before deleting any resources upon which
//	they rely.
//
//	But the destructor also calls into this code to assure that the thread
//	is dead.  This is necessary to assure that the thread is terminated when
//	the app is closing down in a non-standard way (e.g., attempting to cleanly
//	shut down in response to an error condition).
//
void TestThread::DestroyWorkerThread(void)
{
	if (false == m_Thread.IsThreadAlive()) {
		return;
	}

	// Request the thread to stop.
	m_Thread.StopThread();

	// Now wait for the thread to wake up, process the termination request,
	// and exit.
	if (QzSyncWait_Signalled == m_Thread.WaitForClosure()) {
		LogMessage("ThreadFunc is dead");
	}
	else {
		LogErrorMessage("ThreadFunc did not terminate");
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	StaticThreadFunc()
//
//	This is a static method, meaning that the member variables of the class
//	are not accessible.  OS-level thread routines work on function pointers,
//	not classes, so we cannot directly access a class.  However, you can pass
//	in an arbitrary context pointer to pthreads and Win32 threads, so this
//	code passes in the *this pointer as the context.
//
//	All we need to do here is recast the context value and use it to invoke
//	the class's actual thread function.
//
//	Note: This function essentially acts like the main() function for the
//	thread.  When this function call returns, the thread will terminate.
//
U32 TestThread::StaticThreadFunc(void *pContext)
{
	TestThread *p = reinterpret_cast<TestThread*>(pContext);
	return p->ThreadFunc();
}


/////////////////////////////////////////////////////////////////////////////
//
//	ThreadFunc()
//
U32 TestThread::ThreadFunc(void)
{
	for (;;) {
		// Set the time-out duration so the thread will wake up and rotate
		// through all four states once every second.
		U32 result = m_Thread.WaitForEvent(250);

		if (QzSyncWait_Error == result) {
			LogErrorMessage("ThreadFunc wait failed");
			break;
		}

		if (QzSyncWait_StopRequest == result) {
			LogMessage("ThreadFunc detected termination request");
			break;
		}

		// Cycle through four possible counter values.
		m_Counter = (m_Counter + 1) % 4;

		// Set the window's dirty flag so it will know it needs to redraw.
		g_pQzInterface->SetDirtyFlag();
	}

	return 0;
}


