/////////////////////////////////////////////////////////////////////////////
//
//	File: QzLogUtilDlg.h
//
//	$Header: /Projects/QzLogUtilWin/QzLogUtilDlg.cpp  1  2009/9/9 9:44:41a  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#define VC_EXTRALEAN		// Exclude rarely-used stuff from Windows headers

#define _WIN32_WINNT 0x0501

typedef unsigned long    U32;

#include <afxwin.h>         // MFC core and standard components
#include <afxext.h>         // MFC extensions
#include <afxdtctl.h>		// MFC support for Internet Explorer 4 Common Controls
#include "QzLogUtil.h"
#include "QzLogUtilDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


// This header is stored at the beginning of shared memory.  Since shared
// memory is used as a ring buffer, these fields indicate the position of
// the data within the ring buffer.
//
struct QzLogInfo_t
{
	U32 Enable;        // flag used to disable the writing of log messages
	U32 ErrorCount;    // how many error messages have been logged
	U32 MaxByteCount;  // total size of shared memory buffer
	U32 ByteCount;     // total number of bytes in the ring buffer
	U32 ByteOffset;    // offset to the start of the ring
};


/////////////////////////////////////////////////////////////////////////////
//
// CQzLogUtilDlg dialog
//
CQzLogUtilDlg::CQzLogUtilDlg(CWnd* pParent)
	:	CDialog(CQzLogUtilDlg::IDD, pParent),
		m_hFileMapping(NULL),
		m_pFileAddress(NULL),
		m_MappedSize(4 * 1024 * 1024),
		m_EnableLogging(TRUE)
{
	//{{AFX_DATA_INIT(CQzLogUtilDlg)
		// NOTE: the ClassWizard will add member initialization here
	//}}AFX_DATA_INIT
	// Note that LoadIcon does not require a subsequent DestroyIcon in Win32
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
}

void CQzLogUtilDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CQzLogUtilDlg)
		// NOTE: the ClassWizard will add DDX and DDV calls here
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CQzLogUtilDlg, CDialog)
	//{{AFX_MSG_MAP(CQzLogUtilDlg)
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_BN_CLICKED(IDC_QUIT, OnQuit)
	ON_BN_CLICKED(IDC_SETFILE, OnSetFile)
	ON_BN_CLICKED(IDC_SAVE, OnSave)
	ON_BN_CLICKED(IDC_ENABLELOGGING, OnEnableLogging)
	ON_WM_DESTROY()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
//
//	OnInitDialog()
//
BOOL CQzLogUtilDlg::OnInitDialog()
{
	CDialog::OnInitDialog();

	// Set the icon for this dialog.  The framework does this automatically
	// when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
//	SetIcon(m_hIcon, FALSE);		// Set small icon

	// Default permission required to read and write to the shared memory.
	DWORD protection = PAGE_READWRITE;

	// Comment out this line to enable page caching, which delivers a
	// slightly better performance, but which prevents the logger from
	// immediately seeing changes (meaning that the last range of data
	// written to shared memory before a crash may end up being lost --
	// exercise caution when commenting out these flags).
	//
	protection |= SEC_COMMIT | SEC_NOCACHE;

	m_hFileMapping = CreateFileMapping(INVALID_HANDLE_VALUE, NULL, protection, 0, m_MappedSize, L"QzLog");

	// Check whether the buffer already exists.  If so, another app
	// has already created the buffer (possibly due to two instances
	// of the logging utility), so we should not clear out the
	// header at the start of shared memory.
	bool alreadyExists = (GetLastError() == ERROR_ALREADY_EXISTS);

	if (m_hFileMapping != NULL) {
		m_pFileAddress = reinterpret_cast<BYTE*>(MapViewOfFile(m_hFileMapping, FILE_MAP_ALL_ACCESS, 0, 0, m_MappedSize));

		if (m_pFileAddress != NULL) {

			// Only reset the header if the shared buffer did not
			// previously exist.
			if (false == alreadyExists) {
				QzLogInfo_t *pInfo  = reinterpret_cast<QzLogInfo_t*>(m_pFileAddress);
				pInfo->Enable       = m_EnableLogging;
				pInfo->ErrorCount   = 0;
				pInfo->MaxByteCount = m_MappedSize - sizeof(QzLogInfo_t);
				pInfo->ByteCount    = 0;
				pInfo->ByteOffset   = 0;
			}
		}
		else {
			MessageBox(L"Could not map trace information", NULL, MB_ICONEXCLAMATION);
		}
	}
	else {
		MessageBox(L"Shared trace file could not be created", NULL, MB_ICONEXCLAMATION);
	}


	CButton *pButton = (CButton*)GetDlgItem(IDC_ENABLELOGGING);
	pButton->SetCheck(m_EnableLogging);

	CEdit *pEdit = (CEdit*)GetDlgItem(IDC_FILENAME);
	pEdit->SetWindowText(m_FileName);

	SetWindowPos(NULL, m_XCoord, m_YCoord, 0, 0, SWP_NOSIZE | SWP_NOZORDER);

	// return TRUE unless you set the focus to a control
	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
//
void CQzLogUtilDlg::OnDestroy()
{
	RECT rect;
	GetWindowRect(&rect);
	m_XCoord = rect.left;
	m_YCoord = rect.top;

	if (m_pFileAddress != NULL) {
		UnmapViewOfFile(reinterpret_cast<void*>(m_pFileAddress));
		m_pFileAddress = NULL;
	}

	if (m_hFileMapping != NULL) {
		CloseHandle(m_hFileMapping);
		m_hFileMapping = NULL;
	}

	CDialog::OnDestroy();
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnPaint()
//
//	If you add a minimize button to your dialog, you will need the code below
//	to draw the icon.  For MFC applications using the document/view model,
//	this is automatically done for you by the framework.
//
void CQzLogUtilDlg::OnPaint()
{
	if (IsIconic()) {
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, WPARAM(dc.GetSafeHdc()), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else {
		CDialog::OnPaint();
	}
}

/////////////////////////////////////////////////////////////////////////////
//
//	OnQueryDragIcon()
//
//	The system calls this to obtain the cursor to display while the user drags
//	the minimized window.
//
HCURSOR CQzLogUtilDlg::OnQueryDragIcon()
{
	return (HCURSOR) m_hIcon;
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnCancel()
//
//	Override this method to prevent accidentally closing the dialog by
//	hitting the ESC key.
//
void CQzLogUtilDlg::OnCancel()
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnOK()
//
//	Override this, since there is no OK button any longer.  This prevents
//	any lingering MFCisms that try to send OK signals to the dialog.
//
void CQzLogUtilDlg::OnOK()
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnQuit()
//
//	This is now the only way to close the dialog box.
//
void CQzLogUtilDlg::OnQuit()
{
	EndDialog(IDOK);
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnSave()
//
//	When the Save button is clicked, dump the contents of shared memory out
//	to a file.
//
void CQzLogUtilDlg::OnSave()
{
	if (m_pFileAddress == NULL) {
		MessageBox(L"Trace information not accessible", NULL, MB_ICONEXCLAMATION);
	}
	else {
		QzLogInfo_t *pInfo = reinterpret_cast<QzLogInfo_t*>(m_pFileAddress);

		// Note: We absolutely must open the file in binary format instead
		// of the default text format, otherwise fwrite() will happily
		// screw with the binary data when it sees any '\n' in the stream.
		// Since we are writing text data, this is not really critical,
		// but it does mean the file that ends up on disk will not exactly
		// match what was in memory.
		FILE *pFile = _wfopen(m_FileName, L"wb");

		if (pFile != NULL) {
			// Shared memory is treated as a ring buffer.  The header at
			// the start of shared memory tells us what range of data in
			// the buffer contains out logging information.
			U32 byteOffset = pInfo->ByteOffset;
			U32 byteCount  = pInfo->ByteCount;
			U32 maxCount   = pInfo->MaxByteCount;

			// Skip over the header at the start of shared memory to get
			// to the start of the actual log text.
			char *pStart = reinterpret_cast<char*>(m_pFileAddress + sizeof(QzLogInfo_t));

			// Make life easier by allocating a temp buffer, with extra
			// space for a '\0' at the end of the buffer.  This allows
			// us to ignore any logic problems that will arise if a
			// multi-byte UTF-8 symbol is split across the end of the
			// ring buffer.
			char *pTemp = new char[byteCount + 1];

			// If the log data never wrapped around the end of shared
			// memory, we can simply memcpy() the string.
			if (byteCount < maxCount) {
				memcpy(pTemp, pStart, byteCount);
			}

			// Otherwise the data wraps around the end of the ring buffer,
			// so two memcpy's are required to piece the whole block of
			// log data into a continuous buffer.
			else {
				U32 count1 = maxCount - byteOffset;
				U32 count2 = byteOffset;
				memcpy(pTemp, pStart + byteOffset, count1);
				memcpy(pTemp + count1, pStart, count2);
			}

			// Terminate the buffer.  In this case it is not strictly
			// necessary, doing so can avoid any craziness that may arise
			// from trying to view a non-terminated string in a debugger.
			pTemp[byteCount] = '\0';

			U32  skip   = 0;
			bool isUTF8 = false;

			// It's possible that the oldest byte in the buffer is from the
			// middle of a multi-byte sequence.  This will skip any middle
			// bytes, looking for the first 7-bit ASCII byte, or the first
			// byte of a multi-byte sequence.  This should not be more than
			// three bytes into the buffer.  Normally, it should be the very
			// first byte, since almost no multi-byte symbols are used in
			// logging.
			for (U32 i = 0; i < byteCount; ++i) {
				if (0x80 != (0xC0 & pTemp[i])) {
					skip = i;
					break;
				}

				if (0 != (0x80 & pTemp[i])) {
					isUTF8 = true;
				}
			}

			// This three-byte sequence denotes a UTF-8 file.  But we only
			// need to write it if there are any 8-bit symbols.  If all of
			// the text is 7-bit ASCII, we can skip the marker, since WordPad
			// has problems with reading UTF-8 files (it appears to read data
			// in 8KB chunks, and glitches one of the bytes near the end of
			// each chunk).
			//
			if (isUTF8) {
				BYTE marker[3];
				marker[0] = 0xEF;
				marker[1] = 0xBB;
				marker[2] = 0xBF;

				fwrite(marker, 1, 3, pFile);
			}

			// Write all of the log to disk, skipping any partial UTF-8 symbol
			// from the start of the buffer.
			fwrite(pTemp + skip, 1, byteCount - skip, pFile);

			fclose(pFile);

			wchar_t buffer[256];
			swprintf(buffer, L"Log Size: %d bytes", byteCount + 3 - skip);
			MessageBox(buffer);

			delete [] pTemp;
		}
		else {
			wchar_t buffer[MAX_PATH + 64];
			swprintf(buffer, L"Unable to open file \"%s\"", m_FileName);
			MessageBox(buffer, L"Disk Error", MB_ICONEXCLAMATION);
		}

		if (0 != pInfo->ErrorCount) {
			wchar_t buffer[256];

			if (pInfo->ErrorCount > 1) {
				swprintf(buffer, L"There were %d errors recorded in the log.", pInfo->ErrorCount);
			}
			else {
				swprintf(buffer, L"There was 1 error recorded in the log.");
			}

			MessageBox(buffer, L"Errors in Log File", MB_ICONEXCLAMATION);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnSetFile()
//
void CQzLogUtilDlg::OnSetFile()
{
	wchar_t filename[MAX_PATH];
	wcscpy(filename, m_FileName);

	OPENFILENAME data;

	memset(&data, 0, sizeof(data));

	data.lStructSize	= sizeof(data);
	data.lpstrTitle		= L"Select Log File";
	data.lpstrFile		= filename;
	data.nMaxFile		= MAX_PATH;
	data.Flags			= OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
	data.lpstrFilter	= L"Log File (*.log)\0*.log\0";
	data.lpstrDefExt	= L"log";

	if (GetSaveFileName(&data)) {
		wcscpy(m_FileName, filename);

		CEdit *pEdit = (CEdit*)GetDlgItem(IDC_FILENAME);
		pEdit->SetWindowText(filename);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	OnEnableLogging()
//
void CQzLogUtilDlg::OnEnableLogging()
{
	CButton *pButton = (CButton*)GetDlgItem(IDC_ENABLELOGGING);

	if (m_pFileAddress != NULL) {
		QzLogInfo_t *pInfo = reinterpret_cast<QzLogInfo_t*>(m_pFileAddress);
		m_EnableLogging = pInfo->Enable = pButton->GetCheck();
	}
}

