/////////////////////////////////////////////////////////////////////////////
//
//	File: QzVector.h
//
//	$Header: /Projects/Qz/QzVector.h  30  2009/9/7 11:47:32a  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#pragma once


class QzVector
{
public:
	float m_X;
	float m_Y;
	float m_Z;

	QzVector(void) {}
	QzVector(const QzVector_t &v) : m_X(v.X), m_Y(v.Y), m_Z(v.Z) { }
	QzVector(const float x, const float y, const float z) : m_X(x), m_Y(y), m_Z(z) { }

	QzVector(const QzIntVector_t &v)
	{
		m_X = float(v.X) * (1.0f / float(QzFixedPoint_One));
		m_Y = float(v.Y) * (1.0f / float(QzFixedPoint_One));
		m_Z = float(v.Z) * (1.0f / float(QzFixedPoint_One));
	}

	void Set(float x, float y, float z)
	{
		m_X = x;
		m_Y = y;
		m_Z = z;
	}

	QzVector& operator+=(const QzVector &v) {
	    m_X += v.m_X; m_Y += v.m_Y; m_Z += v.m_Z; return *this;
	}

	QzVector& operator-=(const QzVector &v) {
		m_X -= v.m_X; m_Y -= v.m_Y; m_Z -= v.m_Z; return *this;
	}

	QzVector& operator*=(const QzVector &v) {
	    m_X *= v.m_X; m_Y *= v.m_Y; m_Z *= v.m_Z; return *this;
	}

	QzVector& operator/=(const QzVector &v) {
		m_X /= v.m_X; m_Y /= v.m_Y; m_Z /= v.m_Z; return *this;
	}

	QzVector& operator*=(float v) {
		m_X *= v; m_Y *= v; m_Z *= v; return *this;
	}

	QzVector& operator/=(float v) {
		float inv = 1.0f / v; m_X *= inv; m_Y *= inv; m_Z *= inv; return *this;
	}

	QzVector operator+(void) const {
		return *this;
	}

	QzVector operator-(void) const {
		return QzVector(-m_X, -m_Y, -m_Z);
	}

	QzVector operator+(const QzVector &v) const {
		return QzVector(m_X + v.m_X, m_Y + v.m_Y, m_Z + v.m_Z);
	}

	QzVector operator-(const QzVector &v) const {
		return QzVector(m_X - v.m_X, m_Y - v.m_Y, m_Z - v.m_Z);
	}

	QzVector operator*(const QzVector &v) const {
		return QzVector(m_X * v.m_X, m_Y * v.m_Y, m_Z * v.m_Z);
	}

	QzVector operator/(const QzVector &v) const {
		return QzVector(m_X / v.m_X, m_Y / v.m_Y, m_Z / v.m_Z);
	}

	QzVector operator*(float v) const {
		return QzVector(m_X * v, m_Y * v, m_Z * v);
	}

	QzVector operator/(float v) const {
		float inv = 1.0f / v;
		return QzVector(m_X * inv, m_Y * inv, m_Z * inv);
	}

	void MinComponent(const QzVector &v) {
		if (m_X > v.m_X) m_X = v.m_X;
		if (m_Y > v.m_Y) m_Y = v.m_Y;
		if (m_Z > v.m_Z) m_Z = v.m_Z;
	}

	void MaxComponent(const QzVector &v) {
		if (m_X < v.m_X) m_X = v.m_X;
		if (m_Y < v.m_Y) m_Y = v.m_Y;
		if (m_Z < v.m_Z) m_Z = v.m_Z;
	}

	float DistanceSquared(const QzVector &v) const {
		float dx = v.m_X - m_X;
		float dy = v.m_Y - m_Y;
		float dz = v.m_Z - m_Z;

		return ((dx * dx) + (dy * dy) + (dz * dz));
	}

	float Distance(const QzVector &v) const {
		return sqrtf(DistanceSquared(v));
	}

	float LengthSquared(void) {
		return ((m_X * m_X) + (m_Y * m_Y) + (m_Z * m_Z));
	}

	float Length(void) {
		return sqrtf(LengthSquared());
	}

	void  Normalize(void);
	void  Average(const QzVector &v1, const QzVector &v2);
	float DotProduct(const QzVector &v) const;
	void  CrossProduct(const QzVector_t &v1, const QzVector_t &v2, const QzVector_t &v3);
	void  CrossProduct(const QzVector &v1, const QzVector &v2, const QzVector &v3);
	void  CrossProduct(const QzVector &v1, const QzVector &v2);
	float DotProductSegmentByDirection(const QzVector &s0, const QzVector &direction) const;
	float DotProductSegmentByPoints(const QzVector &s0, const QzVector &s1) const;
	float DistanceToSegment(const QzVector &s0, const QzVector &s1) const;
	void  NearestPointOnLine(const QzVector &s0, const QzVector &s1, const QzVector &point);
	bool  NearestPointOnSegment(const QzVector &s0, const QzVector &s1, const QzVector &point);
	bool  NearestPointOnPolygon(const QzVector verts[], const U32 vertCount, const QzVector point);
	bool  IsPointInPolygon(const QzVector verts[], const U32 vertCount);
	bool  IntersectRaySphere(const QzVector &start, const QzVector &dir, const QzVector &sphere, const float radius);
	bool  IntersectRayPolygon(const QzVector &start, const QzVector &dir, const QzVector verts[], const U32 vertCount);
	bool  IntersectRayPlane(const QzVector &start, const QzVector &dir, const QzVector verts[]);
	bool  IntersectRayPlane(const QzVector &start, const QzVector &dir, const QzVector &normal, const QzVector &surface);
	bool  IntersectSegmentPolygon(const QzVector &start, const QzVector &stop, const QzVector verts[], const U32 vertCount);
	bool  IntersectSegmentPlane(const QzVector &start, const QzVector &stop, const QzVector verts[]);
	void  Interpolate(const QzVector &s0, const QzVector &s1, const float interp);

	void  Clear(void)
	{
		m_X = 0.0f;
		m_Y = 0.0f;
		m_Z = 0.0f;
	}

	bool  IsZero(void) const
	{
		return QzFloatIsZero20(m_X) && QzFloatIsZero20(m_Y) && QzFloatIsZero20(m_Z);
	}

	bool  Equal(const QzVector &v) const;

	void  RotateX(float angle);
	void  RotateY(float angle);
	void  RotateZ(float angle);
	void  RotateArbitraryAxis(QzVector axis, float angle);

	QzVector_t GetVector(void) {
		QzVector_t v; v.X = m_X; v.Y = m_Y; v.Z = m_Z;
		return v;
	}

	QzIntVector_t GetIntVector(void) const;
};


