/////////////////////////////////////////////////////////////////////////////
//
//	File: QzRenderer.cpp
//
//	$Header: /Projects/Qz/QzRenderer.cpp  2  2009/9/14 1:39:34p  Lee $
//
//
//	Abstracts rendering behind a class definition.  This keeps all of the
//	platform-specific differences (Windows vs. OS X) in a single place,
//	so that porting to a new platform only requires changing the contents of
//	the CPP file.  It also prevents any other source code from needing to
//	include any platform-specific header files in the rest of the project,
//	protecting the code from any non-standard header behavior (especially
//	the profusion of UNICODE #defines in windows.h).
//
//	This class is very minimalist, with just a little OpenGL used for simple
//	rendering.  I don't use OpenGL's coordinate system, since I normally
//	write code supports rendering by both OpenGL and DirectX, depending on
//	the current platform.  None of that code is used here (it would be
//	overkill for this app), but I keep to my usual coordinate math to avoid
//	introducing confusion.
//
/////////////////////////////////////////////////////////////////////////////


#ifdef _WIN32

#include <windows.h>
#include <GL/gl.h>
extern HDC g_hDC;					// needed for SwapBuffers()
#pragma comment(lib, "opengl32")

#else // OS X

#include <OpenGL/OpenGL.h>
#include <AGL/agl.h>
#endif


#include "QzCommon.h"
#include "QzRenderer.h"
#include "QzTexture.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzRenderer::QzRenderer(void)
	:	m_WindowWidth(16),
		m_WindowHeight(16),
		m_IsCreated(false),
		m_pFontTexture(NULL)
{
	m_pFontTexture = new QzTexture;
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzRenderer::~QzRenderer(void)
{
	SafeDelete(m_pFontTexture);
}


/////////////////////////////////////////////////////////////////////////////
//
//	LogExtensions()
//
//	Dump a list of supported extensions to the log file.  The string itself
//	is one long list of symbols, with only a single space between them.
//	This code splits up the symbols, printing a separate one to each line
//	of the log so they're easier to read.
//
void QzRenderer::LogExtensions(void)
{
	char *pExtensions = const_cast<char*>(reinterpret_cast<const char*>(glGetString(GL_EXTENSIONS)));
	LogMessage("  OpenGL extensions:");

	if (NULL != pExtensions) {
		U32 charCount = U32(strlen(pExtensions));
		char *pDup = new char[charCount+1];
		memcpy(pDup, pExtensions, charCount + 1);

		char *pStart = pDup;

		while ('\0' != *pStart) {
			char *pScan = pStart;
			while (('\0' != *pScan) && (' ' != *pScan)) {
				++pScan;
			}

			if (' ' == *pScan) {
				*pScan = '\0';
				++pScan;
			}

			UtfFormat fmt;
			fmt.AddString(pStart);
			LogMessage("    %1;", fmt);

			pStart = pScan;
		}

		SafeDeleteArray(pDup);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Create()
//
void QzRenderer::Create(void)
{
	m_IsCreated = true;

	m_Fonts.FillTexture(m_pFontTexture, 32);
	TextureBind(m_pFontTexture);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Destroy()
//
void QzRenderer::Destroy(void)
{
	TextureUnbind(m_pFontTexture);

	m_IsCreated = false;
}


/////////////////////////////////////////////////////////////////////////////
//
//	LoadFonts()
//
bool QzRenderer::LoadFonts(const Utf08_t filename[])
{
	return m_Fonts.LoadFontFile(filename);
}


/////////////////////////////////////////////////////////////////////////////
//
//	IsCreated()
//
bool QzRenderer::IsCreated(void)
{
	return m_IsCreated;
}


/////////////////////////////////////////////////////////////////////////////
//
//	SetResolution()
//
void QzRenderer::SetResolution(U32 width, U32 height)
{
	m_WindowWidth  = width;
	m_WindowHeight = height;
}


float g_TextureMatrix[16] =
{
	1.0f,  0.0f, 0.0f, 0.0f,
	0.0f, -1.0f, 0.0f, 0.0f,
	0.0f,  0.0f, 1.0f, 0.0f,
	0.0f,  1.0f, 0.0f, 1.0f
};


/////////////////////////////////////////////////////////////////////////////
//
//	PrepareParallel()
//
void QzRenderer::PrepareParallel(void)
{
	glViewport(0, 0, m_WindowWidth, m_WindowHeight);

	// Fix the projection matrix so that (0,0) is in the upper right corner.
	// This fixes OpenGL's "upside-down" coordinate space to act like DirectX,
	// so the same matrices and coordinates can be computed and used for both
	// OpenGL and DirectX.
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, float(m_WindowWidth), 0, float(m_WindowHeight), 0.0f, 1.0f);
	glTranslatef(0.0f, float(m_WindowHeight) - 0.0f, 0.0f);
	glScalef(1.0f, -1.0f, 1.0f);

	// Also fix the texture matrix so that models use the same U,V coordinates
	// as DirectX, so the same geometry can be used with either DirectX or
	// OpenGL.
//	glMatrixMode(GL_TEXTURE);
//	glLoadIdentity();
//	glLoadMatrixf(g_TextureMatrix);
//	glTranslatef(0.0f, 1.0f, 0.0f);
//	glScalef(1.0f, -1.0f, 1.0f);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Hard-wire this one in place, since it's the mode we'll always use.
	glFrontFace(GL_CW);
}


/////////////////////////////////////////////////////////////////////////////
//
//	ClearFrameBuffer()
//
void QzRenderer::ClearFrameBuffer(void)
{
	glClearDepth(1.0f);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}


/////////////////////////////////////////////////////////////////////////////
//
//	RenderFinish()
//
void QzRenderer::RenderFinish(void)
{
	glFinish();

#ifdef _WIN32
	if (FALSE == SwapBuffers(g_hDC)) {
		LogErrorMessage("SwapBuffers() failed");
	}
#else
#endif
}


/////////////////////////////////////////////////////////////////////////////
//
//	DrawPolygon()
//
void QzRenderer::DrawPolygon(QzVector_t verts[], U32 pointCount)
{
	glBegin(GL_POLYGON);

	for (U32 i = 0; i < pointCount; ++i) {
		glVertex3f(verts[i].X, verts[i].Y, verts[i].Z);
	}

	glEnd();
}


/////////////////////////////////////////////////////////////////////////////
//
//	DrawLineList()
//
void QzRenderer::DrawLineList(QzVector_t verts[], U32 lineCount)
{
	glBegin(GL_LINES);

	for (U32 i = 0; i < lineCount; ++i) {
		glVertex3f(verts[i*2  ].X, verts[i*2  ].Y, verts[i*2  ].Z);
		glVertex3f(verts[i*2+1].X, verts[i*2+1].Y, verts[i*2+1].Z);
	}

	glEnd();
}


/////////////////////////////////////////////////////////////////////////////
//
//	DrawPointList()
//
void QzRenderer::DrawPointList(QzVector_t verts[], U32 pointCount)
{
	glBegin(GL_POINTS);

	for (U32 i = 0; i < pointCount; ++i) {
		glVertex3f(verts[i].X, verts[i].Y, verts[i].Z);
	}

	glEnd();
}


/////////////////////////////////////////////////////////////////////////////
//
//	FontRender()
//
void QzRenderer::FontRender(U32 fontID, S32 x, S32 y, const Utf08_t strg[])
{
	Utf32_t dup[256];

	U32 charCount = UtfConvert08to32(dup, ArraySize(dup), strg);

	if (charCount > 0) {
		glBindTexture(GL_TEXTURE_2D, m_pFontTexture->m_hTexture);
//		glActiveTexture(GL_TEXTURE0);
		glEnable(GL_TEXTURE_2D);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		m_Fonts.AssemblePixels(x, y, 0xFFFFFF, fontID, dup, charCount);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_BLEND);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	TextureBind()
//
U32 QzRenderer::TextureBind(QzTexture *pTexture)
{
	GLuint hTex = 0;

	if (NULL != pTexture) {
		glGenTextures(1, &hTex);
		glBindTexture(GL_TEXTURE_2D, hTex);
		glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, pTexture->m_Width, pTexture->m_Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, pTexture->m_pBuffer);

		pTexture->m_hTexture = hTex;
	}

	return hTex;
}


/////////////////////////////////////////////////////////////////////////////
//
//	TextureUnbind()
//
void QzRenderer::TextureUnbind(QzTexture *pTexture)
{
	if ((NULL != pTexture) && (0 != pTexture->m_hTexture)) {
		GLuint hTex = pTexture->m_hTexture;

		glDeleteTextures(1, &hTex);

		pTexture->m_hTexture = 0;
	}
}






