/////////////////////////////////////////////////////////////////////////////
//
//	File: QzRectSorter.cpp
//
//	$Header: /TS/TsGui/QzRectSorter.cpp  7  2009/9/7 3:23:43p  Lee $
//
//
//	This is a utility class used to pack sub-rectangles into a larger area.
//	It is used to pack individual lightmaps, textures, and font glyphs into
//	a larger buffer, assuring that each rectanle is assigned a unique region
//	of space so no two rectangles overlap.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzRectSorter.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzRectSorter::QzRectSorter(void)
	:	m_pLeft(NULL),
		m_pRight(NULL)
{
	m_Position.pContext = NULL;
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzRectSorter::~QzRectSorter(void)
{
	SafeDelete(m_pLeft);
	SafeDelete(m_pRight);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Initialize()
//
//	Destroys any sub-rectangles from a previous pass, and resets the current
//	region to the given size.
//
void QzRectSorter::Initialize(U32 width, U32 height)
{
	m_Position.Left		= 0;
	m_Position.Top		= 0;
	m_Position.Width	= width;
	m_Position.Height	= height;
	m_Position.pSort	= NULL;
	m_Position.pContext	= NULL;

	SafeDelete(m_pLeft);
	SafeDelete(m_pRight);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Insert()
//
//	Performs a recursive insertion of a new rectangular region within the
//	parent region.  If no sub-region is large enough to contain the new
//	region, it returns NULL.
//
//	The code is recursive, attempting to split the current region into two
//	separate regions, one of which is assigned to the new region, and the
//	other becoming a free chunk of space.
//
QzRectSorter* QzRectSorter::Insert(const QzRectSortPosition_t &position)
{
	// If this node is not a leaf, try adding the rectangle to each of the
	// children in turn.
	if (NULL != m_pLeft) {
		QzRectSorter *pNew = m_pLeft->Insert(position);

		if (NULL != pNew) {
			return pNew;
		}

		return m_pRight->Insert(position);
	}

	// Otherwise, this is a leaf.  If it already contains an image, nothing
	// can be inserted here.
	else if (NULL != m_Position.pContext) {
		return NULL;
	}

	// Test whether there is enough space for the rectangle.  If the rect is too
	// large, fail the insertion attempt.
	else if ((position.Width > m_Position.Width) || (position.Height > m_Position.Height)) {
		return NULL;
	}

	// This is an empty leaf, and the rectangle can be inserted.
	//
	// If the area is large enough, split it into two leaves, storing the
	// rectangle in one and leaving the other free.
	//
	// If there is not enough extra space, simply insert the rectangle here
	// and make this a terminal leaf, in which case nothing more can be
	// added below this part of the tree.
	//
	else if (((position.Width + 1) >= m_Position.Width) && ((position.Height + 1) >= m_Position.Height)) {
		m_Position.pContext = position.pContext;

		return this;
	}

	else {
		QzAssert(NULL == m_pLeft);
		QzAssert(NULL == m_pRight);

		m_pLeft  = new QzRectSorter;
		m_pRight = new QzRectSorter;

		U32 extraWidth  = m_Position.Width  - position.Width;
		U32 extraHeight = m_Position.Height - position.Height;

		if (extraWidth > extraHeight) {
			m_pLeft->m_Position.Width   = position.Width;
			m_pLeft->m_Position.Height	= m_Position.Height;
			m_pLeft->m_Position.Left	= m_Position.Left;
			m_pLeft->m_Position.Top		= m_Position.Top;

			m_pRight->m_Position.Width	= extraWidth;
			m_pRight->m_Position.Height	= m_Position.Height;
			m_pRight->m_Position.Left	= m_Position.Left + position.Width;
			m_pRight->m_Position.Top	= m_Position.Top;
		}
		else {
			m_pLeft->m_Position.Width   = m_Position.Width;
			m_pLeft->m_Position.Height	= position.Height;
			m_pLeft->m_Position.Left	= m_Position.Left;
			m_pLeft->m_Position.Top		= m_Position.Top;

			m_pRight->m_Position.Width	= m_Position.Width;
			m_pRight->m_Position.Height	= extraHeight;
			m_pRight->m_Position.Left	= m_Position.Left;
			m_pRight->m_Position.Top	= m_Position.Top + position.Height;
		}

		// Note the final bit of recursion: We've just subdivided the current
		// region on one axis, but both resulting regions are still free.
		// We now recurse along the left sub-tree, which should be just large
		// enough in one dimension to hold the new region.  This will cause
		// the left sub-region to be split (if possible) along the other
		// dimension, so the new sub-region only takes up just enough space
		// required by it.  Any remaining space then becomes another free
		// chunk that may be used in a future allocation.

		return m_pLeft->Insert(position);
	}
}



