//////////////////////////////////////////////////////////////////////////////
//
//	File: QzQuat.cpp
//
//	$Header: /TS/TsGui/QzQuat.cpp  6  2009/9/7 2:15:45p  Lee $
//
//////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzMatrix3x3.h"
#include "QzMatrix4x4.h"
#include "QzQuat.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


//////////////////////////////////////////////////////////////////////////////
//
//	Multiply()
//
void QzQuat::Multiply(const QzQuat &a, const QzQuat &b)
{
	m_X = (a.m_X * b.m_W) + (a.m_W * b.m_X) + (a.m_Z * b.m_Y) - (a.m_Y * b.m_Z);
	m_Y = (a.m_Y * b.m_W) - (a.m_Z * b.m_X) + (a.m_W * b.m_Y) + (a.m_X * b.m_Z);
	m_Z = (a.m_Z * b.m_W) + (a.m_Y * b.m_X) - (a.m_X * b.m_Y) + (a.m_W * b.m_Z);
	m_W = (a.m_W * b.m_W) - (a.m_X * b.m_X) - (a.m_Y * b.m_Y) - (a.m_Z * b.m_Z);
}


//////////////////////////////////////////////////////////////////////////////
//
//	ConvertToMatrix()
//
void QzQuat::ConvertToMatrix(QzMatrix3x3 &matrix) const
{
	// calculate coefficients
	float x2 = m_X + m_X;
	float y2 = m_Y + m_Y;
	float z2 = m_Z + m_Z;
	float xx = m_X * x2;
	float xy = m_X * y2;
	float xz = m_X * z2;
	float yy = m_Y * y2;
	float yz = m_Y * z2;
	float zz = m_Z * z2;
	float wx = m_W * x2;
	float wy = m_W * y2;
	float wz = m_W * z2;

	matrix.m_Matrix.c3x3[0][0] = 1.0f - (yy + zz);
	matrix.m_Matrix.c3x3[0][1] = xy + wz;
	matrix.m_Matrix.c3x3[0][2] = xz - wy;

	matrix.m_Matrix.c3x3[1][0] = xy - wz;
	matrix.m_Matrix.c3x3[1][1] = 1.0f - (xx + zz);
	matrix.m_Matrix.c3x3[1][2] = yz + wx;

	matrix.m_Matrix.c3x3[2][0] = xz + wy;
	matrix.m_Matrix.c3x3[2][1] = yz - wx;
	matrix.m_Matrix.c3x3[2][2] = 1.0f - (xx + yy);
}


//////////////////////////////////////////////////////////////////////////////
//
//	ConvertToMatrix()
//
void QzQuat::ConvertToMatrix(QzMatrix4x4 &matrix) const
{
	// calculate coefficients
	float x2 = m_X + m_X;
	float y2 = m_Y + m_Y;
	float z2 = m_Z + m_Z;
	float xx = m_X * x2;
	float xy = m_X * y2;
	float xz = m_X * z2;
	float yy = m_Y * y2;
	float yz = m_Y * z2;
	float zz = m_Z * z2;
	float wx = m_W * x2;
	float wy = m_W * y2;
	float wz = m_W * z2;

	matrix.m_Matrix.c4x4[0][0] = 1.0f - (yy + zz);
	matrix.m_Matrix.c4x4[0][1] = xy + wz;
	matrix.m_Matrix.c4x4[0][2] = xz - wy;
	matrix.m_Matrix.c4x4[0][3] = 0.0f;

	matrix.m_Matrix.c4x4[1][0] = xy - wz;
	matrix.m_Matrix.c4x4[1][1] = 1.0f - (xx + zz);
	matrix.m_Matrix.c4x4[1][2] = yz + wx;
	matrix.m_Matrix.c4x4[1][3] = 0.0f;

	matrix.m_Matrix.c4x4[2][0] = xz + wy;
	matrix.m_Matrix.c4x4[2][1] = yz - wx;
	matrix.m_Matrix.c4x4[2][2] = 1.0f - (xx + yy);
	matrix.m_Matrix.c4x4[2][3] = 0.0f;

	matrix.m_Matrix.c4x4[3][0] = 0.0f;
	matrix.m_Matrix.c4x4[3][1] = 0.0f;
	matrix.m_Matrix.c4x4[3][2] = 0.0f;
	matrix.m_Matrix.c4x4[3][3] = 1.0f;
}


//////////////////////////////////////////////////////////////////////////////
//
//	ConvertToAxisAngle()
//
//	Warning: The quaternion needs to be normalized before creating the
//	axis-angle.  This code will not explicitly normalize the quaternion,
//	since that would potentially change the quaterion (and would waste cycles
//	if the quat is already normalized).
//
void QzQuat::ConvertToAxisAngle(QzVector &axis, float &angle)
{
	angle = acosf(m_W) * 2.0f;
	float sinA = sqrtf(1.0f - (m_W * m_W));
	if (fabsf(sinA) > 0.0005f) {
		axis.m_X = m_X / sinA;
		axis.m_Y = m_Y / sinA;
		axis.m_Z = m_Z / sinA;
	}
	else {
		axis.m_X = m_X;
		axis.m_Y = m_Y;
		axis.m_Z = m_Z;
	}
}


//////////////////////////////////////////////////////////////////////////////
//
//	MakeFromAxisAngle()
//
void QzQuat::MakeFromAxisAngle(const QzVector &axis, float angle)
{
	float s = sinf(angle / 2.0f);

	m_X = s * axis.m_X;
	m_Y = s * axis.m_Y;
	m_Z = s * axis.m_Z;
	m_W = cosf(angle / 2.0f);
}


//////////////////////////////////////////////////////////////////////////////
//
//	MakeFromVectors()
//
//	Creates a quat that rotates the src vector to the dst vector.
//
void QzQuat::MakeFromVectors(QzVector src, QzVector dst)
{
	src.Normalize();
	dst.Normalize();

	if (src.Equal(dst)) {
		MakeIdentity();
	}
	else {
		float cosTheta = dst.DotProduct(src);
		QzVector axis;

		axis.CrossProduct(dst, src);
		if (axis.IsZero()) {
			axis = QzVector(0.0f, 0.0f, 1.0f);
		}
		else {
			axis.Normalize();
		}
		float angle = acosf(cosTheta);
		MakeFromAxisAngle(axis, -angle);
	}
}


//////////////////////////////////////////////////////////////////////////////
//
//	ApplyRotation()
//
void QzQuat::ApplyRotation(const QzVector &rotation)
{
	QzQuat q, r;
	q.m_X = rotation.m_X;
	q.m_Y = rotation.m_Y;
	q.m_Z = rotation.m_Z;
	q.m_W = 0.0f;

	r.Multiply(q, *this);

	m_X += r.m_X * 0.5f;
	m_Y += r.m_Y * 0.5f;
	m_Z += r.m_Z * 0.5f;
	m_W += r.m_W * 0.5f;

	Normalize();
}


//////////////////////////////////////////////////////////////////////////////
//
//	LinearInterpolation()
//
void QzQuat::LinearInterpolation(const QzQuat &a, const QzQuat &b, const float interp)
{
	// If the two quats are pointing in the same direction, we can LERP
	// between them normally.
	if (a.DotProduct(b) > 0.0f) {
		m_X = (a.m_X * (1.0f - interp)) + (b.m_X * interp);
		m_Y = (a.m_Y * (1.0f - interp)) + (b.m_Y * interp);
		m_Z = (a.m_Z * (1.0f - interp)) + (b.m_Z * interp);
		m_W = (a.m_W * (1.0f - interp)) + (b.m_W * interp);
	}
	// However, if they are pointing in opposite directions, LERP can cause
	// the result to "go the long way around".  Negate one of the quats and
	// interpolate to LERP along the shorter path.
	else {
		m_X = (a.m_X * (interp - 1.0f)) + (b.m_X * interp);
		m_Y = (a.m_Y * (interp - 1.0f)) + (b.m_Y * interp);
		m_Z = (a.m_Z * (interp - 1.0f)) + (b.m_Z * interp);
		m_W = (a.m_W * (interp - 1.0f)) + (b.m_W * interp);
	}

	Normalize();
}


//////////////////////////////////////////////////////////////////////////////
//
//	SphericalInterpolation()
//
void QzQuat::SphericalInterpolation(const QzQuat &a, const QzQuat &b, const float interp)
{
	// The dot product returns the cosine of the angle between the quats.
	float dot = a.DotProduct(b);
	float scale0, scale1;
/*
	// Always interpolate along the shorter arc
	if (dot > 0.01f) {
		// If the angle is not close to zero, it's safe to use SLERP.
		if ((1.0 - dot) > 0.01f) {
			float omega = acosf(dot);
			float sinom = sinf(omega);
			scale0 = sinf((1.0f - interp) * omega) / sinom;
			scale1 = sinf(interp * omega) / sinom;
		}
		// Otherwise use linear interpolation for small angles.
		else {
			scale0 = 1.0f - interp;
			scale1 = interp;
		}
	}
	else {
		// If the angle is not close to zero, it's safe to use SLERP.
		if ((1.0 + dot) > 0.01f) {
			float omega = acosf(-dot);
			float sinom = sinf(omega);
			// FIXME: verify whether these should be negated -- it matches the
			// results from DirectX, but may be rotating in the wrong direction
			scale0 = -sinf((interp - 1.0f) * omega) / sinom;
			scale1 = -sinf(interp * omega) / sinom;
		}
		// Otherwise use linear interpolation for small angles.
		else {
			scale0 = interp - 1.0f;
			scale1 = interp;
		}
	}

	m_X = (scale0 * a.m_X) + (scale1 * b.m_X);
	m_Y = (scale0 * a.m_Y) + (scale1 * b.m_Y);
	m_Z = (scale0 * a.m_Z) + (scale1 * b.m_Z);
	m_W = (scale0 * a.m_W) + (scale1 * b.m_W);
*/

	// Check bounds conditions.  SLERP is normally safe, but we need special
	// cases for quats that are almost identical (in which case we need to
	// LERP to avoid disivion by zero), and when the quats are 180 degrees
	// away from one another (where we need to SLERP with the perpendicular
	// quaterion).
	if ((1.0f + dot) > 0.001f) {
		float omega, sinom;

		// Check the magnitude of the difference.  If the directions are
		// different enough, it is safe to SLERP.
		if ((1.0f - dot) > 0.001f) {
			// YES, DO A SLERP
			omega  = acosf(dot);
			sinom  = sinf(omega);
			scale0 = sinf((1.0f - interp) * omega) / sinom;
			scale1 = sinf(interp * omega) / sinom;
		}

		// Otherwise we need to LERP to avoid division by zero.
		else {
			scale0 = 1.0f - interp;
			scale1 = interp;
		}

		m_X = (scale0 * a.m_X) + (scale1 * b.m_X);
		m_Y = (scale0 * a.m_Y) + (scale1 * b.m_Y);
		m_Z = (scale0 * a.m_Z) + (scale1 * b.m_Z);
		m_W = (scale0 * a.m_W) + (scale1 * b.m_W);
	}

	// Otherwise, the quats are almost opposites.  Avoid division-by-zero
	// by computing a perpendicular quat, then SLERP in that direction.
	else {
		// Note the x/y and z/w swapping: this is computing a perpendicular
		// quaternion.
		m_X = -b.m_Y;
		m_Y =  b.m_X;
		m_Z = -b.m_W;
		m_W =  b.m_Z;

		scale0 = sinf((1.0f - interp) * (c_PI / 2.0f));
		scale1 = sinf(interp * (c_PI / 2.0f));

		m_X = (scale0 * a.m_X) + (scale1 * m_X);
		m_Y = (scale0 * a.m_Y) + (scale1 * m_Y);
		m_Z = (scale0 * a.m_Z) + (scale1 * m_Z);
		m_W = (scale0 * a.m_W) + (scale1 * m_W);
	}
}


//////////////////////////////////////////////////////////////////////////////
//
//	Print()
//
void QzQuat::Print(void)
{
	printf("quat: x = %f, y = %f, z = %f, w = %f\n", m_X, m_Y, m_Z, m_W);
}


