/////////////////////////////////////////////////////////////////////////////
//
//	File: QzOggDecoder.cpp
//
//	$Header: /Projects/Qz/QzOggDecoder.cpp  7  2009/9/14 12:34:06a  Lee $
//
//
//	Wrapper class around the Ogg Vorbis file reading routines.  This is based
//	around having the compressed data sitting in a buffer in memory.  Ogg's
//	callback interface is used to read data from the buffer.  This allows the
//	audio data to be stored in a custom resource file, and be decompressed
//	from memory.
//
//	NOTE: To build with Xcode, the Ogg.framework and Vorbis.framework must
//	be installed on the system.  These can be found at http://www.xiph.org/
//	as pre-built installs.  These tend to end up in /Library/Frameworks/
//	instead of /System/Library/Frameworks/ so some searching around may be
//	required before you can get them added to the project correctly.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzOggDecoder.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


// Link in the DLL containing all of the Ogg routines.  This is a custom
// build that packs all of the needed Ogg/Vorbis DLLs into a single DLL to
// make shipping things easier.
#ifdef _WIN32
#pragma comment(lib, "libvorbis")
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	OggErrorToString()
//
//	Remap common Ogg error codes to strings for debugging/logging messages.
//
char* OggErrorToString(int code)
{
	switch(code) {
		case OV_EREAD:		return "Ogg: data read";
		case OV_EFAULT:		return "Ogg: internal fault";
		case OV_EIMPL:		return "Ogg: implementation";
		case OV_ENOTVORBIS:	return "Ogg: not vorbis data";
		case OV_EBADHEADER:	return "Ogg: invalid vorbis header";
		case OV_EVERSION:	return "Ogg: vorbis version";
		case OV_ENOTAUDIO:	return "Ogg: not audio data";
		case OV_EBADPACKET:	return "Ogg: bad packet";
		case OV_EBADLINK:	return "Ogg: bad link";
		case OV_ENOSEEK:	return "Ogg: no seek";
	}

	return "Ogg: unknown error";
}


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzOggDecoder::QzOggDecoder(void)
	:	m_IsOpen(false)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzOggDecoder::~QzOggDecoder(void)
{
	Stop();
}


/////////////////////////////////////////////////////////////////////////////
//
//	Start()
//
bool QzOggDecoder::Start(U08 *pData, U32 byteCount)
{
	Stop();

	m_pData     = pData;
	m_ByteCount = byteCount;
	m_Offset    = 0;

	// Setup the callback functions that are needed to implement a memory
	// file interface.  These callbacks will be used to redirect the calls
	// into the actual Data...() functions used by Ogg to access the buffer.
	ov_callbacks callbacks;
	callbacks.read_func  = CallbackRead;
	callbacks.seek_func  = CallbackSeek;
	callbacks.close_func = CallbackClose;
	callbacks.tell_func  = CallbackTell;

	// Open the file data in the memory buffer, using these callbacks.
	// This will immediately do some seeks through the buffer to extract
	// some of information about the file, but the operation completes
	// very fast for the files tested.  It's assumed this is safe to do
	// when initializing the buffer.
	S32 result = ov_open_callbacks(this, &m_Context, NULL, 0, callbacks);

	if (0 != result) {
		UtfFormat fmt;
		fmt.AddString(OggErrorToString(result));
		LogErrorMessage("ov_open_callbacks() %1;", fmt);

		return false;
	}

	m_IsOpen = true;

	// Fetch a pointer to the info struct so we can find out the sample
	// rate and whether the file is mono or stereo.
	vorbis_info *pInfo = ov_info(&m_Context, 0);

	if (NULL != pInfo) {
		m_ChannelCount = pInfo->channels;
		m_SampleRate   = pInfo->rate;
	}
	else {
		LogErrorMessage("ov_info() returned NULL pointer");
		return false;
	}

	// A completely different function is required to determine the duration
	// of the file.
	m_SliceCount = U32(ov_pcm_total(&m_Context, -1));

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Loop()
//
//	Forces Ogg to return to the start of the sound file, allowing it to
//	decode the contents of the file a second time.
//
void QzOggDecoder::Loop(void)
{
	if (m_IsOpen) {
		ov_raw_seek(&m_Context, 0);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Read()
//
//	Attempts to extract the requested number of slices from the decoder.
//	It will return one or two samples per slice, depending upon whether the
//	data is mono or stereo.
//
//	Returns the total number of slices extracted.  This is zero when the file
//	has been completed (or an error occurs).
//
U32 QzOggDecoder::Read(S16 samples[], U32 sliceCount)
{
	if (false == m_IsOpen) {
		return 0;
	}

	// The bitstream value is simply a status indicator for which bitstream
	// is currently being decoded.  The value is not needed for anything by
	// this code.
	int bitstream     = 0;
	U32 offset        = 0;
	U32 requestedSize = sliceCount * m_ChannelCount * sizeof(S16);

	char* pBuffer = reinterpret_cast<char*>(samples);

	// This may need to repeat several times to fill up the output buffer.
	// Ogg usually decodes data in 4KB blocks, which probably won't be enough
	// to fill the buffer in a single call to ov_read().
	//
	while (offset < requestedSize) {
		// Parameters 4, 5, and 6 are control parameters.
		//	param 4: 0 = little-endian, 1 = big-endian
		//	param 5: 1 = 8-bit samples, 2 = 16-bit samples
		//	param 6: 0 = unsigned,      1 = signed
#ifdef IS_BIG_ENDIAN
		S32 byteCount = ov_read(&m_Context, pBuffer + offset, requestedSize - offset, 1, 2, 1, &bitstream);
#else
		S32 byteCount = ov_read(&m_Context, pBuffer + offset, requestedSize - offset, 0, 2, 1, &bitstream);
#endif

		// Trap errors, and always return zero to trick the caller into
		// thinking the file is complete so it won't attempt to decode any
		// more data (note that this won't necessarily do any good if the
		// file is being looped, and could potentially produce infinite
		// loops if it keeps calling Loop() and Read()).
		if (byteCount < 0) {
			UtfFormat fmt;
			fmt.AddString(OggErrorToString(byteCount));
			LogErrorMessage("ov_read() %1;", fmt);
			return 0;
		}

		// End of file.
		if (0 == byteCount) {
			break;
		}

		offset += byteCount;
	}

	return offset / (m_ChannelCount * sizeof(S16));
}


/////////////////////////////////////////////////////////////////////////////
//
//	Stop()
//
void QzOggDecoder::Stop(void)
{
	if (m_IsOpen) {
		ov_clear(&m_Context);

		m_IsOpen = false;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	CallbackRead()
//	CallbackSeek()
//	CallbackClose()
//	CallbackTell()
//
//	These are callback functions, used for forward calls from Ogg into the
//	appropriate decoder object.
//
size_t QzOggDecoder::CallbackRead(void *pDest, size_t size, size_t sliceCount, void *pContext)
{
	return reinterpret_cast<QzOggDecoder*>(pContext)->DataRead(pDest, U32(size), U32(sliceCount));
}

int QzOggDecoder::CallbackSeek(void *pContext, ogg_int64_t offset, int origin)
{
	return reinterpret_cast<QzOggDecoder*>(pContext)->DataSeek(U32(offset), origin);
}

int QzOggDecoder::CallbackClose(void *pContext)
{
	return reinterpret_cast<QzOggDecoder*>(pContext)->DataClose();
}

S32 QzOggDecoder::CallbackTell(void *pContext)
{
	return reinterpret_cast<QzOggDecoder*>(pContext)->DataTell();
}


/////////////////////////////////////////////////////////////////////////////
//
//	DataTell()
//
//	Equivalent to ftell().
//
U32 QzOggDecoder::DataTell(void)
{
	return m_Offset;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DataSeek()
//
//	Equivalent to fseek().
//
int QzOggDecoder::DataSeek(U32 offset, int origin)
{
	switch (origin) {
		case SEEK_CUR:
			m_Offset += offset;

			if (m_Offset > m_ByteCount) {
				m_Offset = m_ByteCount;
			}
			break;

		case SEEK_SET:
			if (offset < m_ByteCount) {
				m_Offset = offset;
			}
			else {
				m_Offset = m_ByteCount;
			}
			break;

		case SEEK_END:
			if (offset < m_ByteCount) {
				m_Offset = m_ByteCount - offset;
			}
			else {
				m_Offset = 0;
			}
			break;

		default:
			return -1;
	}

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DataRead()
//
//	Equivalent to fread().  Note that care needs to be taken with the size
//	and count fields, since Ogg will read past the end of the file, so the
//	code must properly scale back the amount of data read and return the
//	correct count value.
//
U32 QzOggDecoder::DataRead(void *pDest, U32 dataSize, U32 dataCount)
{
	U32 byteCount = dataSize * dataCount;

	// Ogg will happily read past the end of the file when starting up.
	// Do a range check to avoid running past the end of the buffer.
	// Make certain that the byte count to read is a multiple of the
	// requested element size.
	if ((m_Offset + byteCount) > m_ByteCount) {
		byteCount = ((m_ByteCount - m_Offset) / dataSize) * dataSize;
	}

	if (byteCount > 0) {
		memcpy(pDest, m_pData + m_Offset, byteCount);
	}

	m_Offset += byteCount;

	// Make certain that the value returned is the number of data units
	// that were read, not the total number of bytes.
	return byteCount / dataSize;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DataClose()
//
//	Equivalent to fclose().
//
U32 QzOggDecoder::DataClose(void)
{
	return 0;
}


