/////////////////////////////////////////////////////////////////////////////
//
//	File: QzMatrix3x3.cpp
//
//	$Header: /TS/TsGui/QzMatrix3x3.cpp  2  2009/9/8 4:11:56p  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzMatrix3x3.h"
#include "QzQuat.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	IsIdentity()
//
bool QzMatrix3x3::IsIdentity(void)
{
	float *f = m_Matrix.cFlat;

	return (QzEqualFloats20(1.0f, f[0]) &&
			QzEqualFloats20(1.0f, f[4]) &&
			QzEqualFloats20(1.0f, f[8]) &&
			QzFloatIsZero20(f[1]) &&
			QzFloatIsZero20(f[2]) &&
			QzFloatIsZero20(f[3]) &&
			QzFloatIsZero20(f[5]) &&
			QzFloatIsZero20(f[6]) &&
			QzFloatIsZero20(f[7]));
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeIdentity()
//
void QzMatrix3x3::MakeIdentity(void)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = 1.0f;
	m_Matrix.c3x3[1][1] = 1.0f;
	m_Matrix.c3x3[2][2] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateX()
//
void QzMatrix3x3::MakeRotateX(float angle)
{
	MakeZero();

	float sinTheta = sinf(angle);
	float cosTheta = cosf(angle);

	m_Matrix.c3x3[0][0] =  1.0f;
	m_Matrix.c3x3[1][1] =  cosTheta;
	m_Matrix.c3x3[1][2] =  sinTheta;
	m_Matrix.c3x3[2][1] = -sinTheta;
	m_Matrix.c3x3[2][2] =  cosTheta;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateY()
//
void QzMatrix3x3::MakeRotateY(float angle)
{
	MakeZero();

	float sinTheta = sinf(angle);
	float cosTheta = cosf(angle);

	m_Matrix.c3x3[0][0] =  cosTheta;
	m_Matrix.c3x3[0][2] = -sinTheta;
	m_Matrix.c3x3[1][1] =  1.0f;
	m_Matrix.c3x3[2][0] =  sinTheta;
	m_Matrix.c3x3[2][2] =  cosTheta;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateZ()
//
void QzMatrix3x3::MakeRotateZ(float angle)
{
	MakeZero();

	float sinTheta = sinf(angle);
	float cosTheta = cosf(angle);

	m_Matrix.c3x3[0][0] =  cosTheta;
	m_Matrix.c3x3[0][1] =  sinTheta;
	m_Matrix.c3x3[1][0] = -sinTheta;
	m_Matrix.c3x3[1][1] =  cosTheta;
	m_Matrix.c3x3[2][2] =  1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateArbitraryAxis()
//
void QzMatrix3x3::MakeRotateArbitraryAxis(float x, float y, float z, float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);
	float t = 1.0f - c;

	m_Matrix.c3x3[0][0] = (t * x * x) + (c    );
	m_Matrix.c3x3[0][1] = (t * y * x) + (s * z);
	m_Matrix.c3x3[0][2] = (t * z * x) - (s * y);

	m_Matrix.c3x3[1][0] = (t * x * y) - (s * z);
	m_Matrix.c3x3[1][1] = (t * y * y) + (c    );
	m_Matrix.c3x3[1][2] = (t * z * y) + (s * x);

	m_Matrix.c3x3[2][0] = (t * x * z) + (s * y);
	m_Matrix.c3x3[2][1] = (t * y * z) - (s * x);
	m_Matrix.c3x3[2][2] = (t * z * z) + (c    );
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeScale()
//
void QzMatrix3x3::MakeScale(QzVector vec)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = vec.m_X;
	m_Matrix.c3x3[1][1] = vec.m_Y;
	m_Matrix.c3x3[2][2] = vec.m_Z;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeScale()
//
void QzMatrix3x3::MakeScale(float factor)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = factor;
	m_Matrix.c3x3[1][1] = factor;
	m_Matrix.c3x3[2][2] = factor;
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorConeX()
//
//	Cone with its principal axis along the X-axis.
//
//	If the base is at the origin, the center of mass is (height / 4.0f) along
//	the X-axis.
//
void QzMatrix3x3::TensorConeX(float mass, float height, float radius)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = ((3.0f / 10.0f) * mass * radius * radius);
	m_Matrix.c3x3[1][1] = ((3.0f / 80.0f) * mass * height * height) + ((3.0f / 20.0f) * mass * radius * radius);
	m_Matrix.c3x3[2][2] = ((3.0f / 80.0f) * mass * height * height) + ((3.0f / 20.0f) * mass * radius * radius);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorConeY()
//
//	Cone with its principal axis along the Y-axis.
//
//	If the base is at the origin, the center of mass is (height / 4.0f) along
//	the Y-axis.
//
void QzMatrix3x3::TensorConeY(float mass, float height, float radius)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = ((3.0f / 80.0f) * mass * height * height) + ((3.0f / 20.0f) * mass * radius * radius);
	m_Matrix.c3x3[1][1] = ((3.0f / 10.0f) * mass * radius * radius);
	m_Matrix.c3x3[2][2] = ((3.0f / 80.0f) * mass * height * height) + ((3.0f / 20.0f) * mass * radius * radius);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorConeZ()
//
//	Cone with its principal axis along the Z-axis.
//
//	If the base is at the origin, the center of mass is (height / 4.0f) along
//	the Z-axis.
//
void QzMatrix3x3::TensorConeZ(float mass, float height, float radius)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = ((3.0f / 80.0f) * mass * height * height) + ((3.0f / 20.0f) * mass * radius * radius);
	m_Matrix.c3x3[1][1] = ((3.0f / 80.0f) * mass * height * height) + ((3.0f / 20.0f) * mass * radius * radius);
	m_Matrix.c3x3[2][2] = ((3.0f / 10.0f) * mass * radius * radius);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorCuboid()
//
void QzMatrix3x3::TensorCuboid(float mass, float x, float y, float z)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = (mass * ((y * y) + (z * z))) / 12.0f;
	m_Matrix.c3x3[1][1] = (mass * ((x * x) + (z * z))) / 12.0f;
	m_Matrix.c3x3[2][2] = (mass * ((x * x) + (y * y))) / 12.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorCylinderX()
//
//	Cylinder with its principal axis along the X-axis.
//
void QzMatrix3x3::TensorCylinderX(float mass, float height, float radius)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = ((mass * radius * radius) / 2.0f);
	m_Matrix.c3x3[1][1] = ((mass * height * height) / 12.0f) + ((mass * radius * radius) / 4.0f);
	m_Matrix.c3x3[2][2] = ((mass * height * height) / 12.0f) + ((mass * radius * radius) / 4.0f);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorCylinderY()
//
//	Cylinder with its principal axis along the Y-axis.
//
void QzMatrix3x3::TensorCylinderY(float mass, float height, float radius)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = ((mass * height * height) / 12.0f) + ((mass * radius * radius) / 4.0f);
	m_Matrix.c3x3[1][1] = ((mass * radius * radius) / 2.0f);
	m_Matrix.c3x3[2][2] = ((mass * height * height) / 12.0f) + ((mass * radius * radius) / 4.0f);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorCylinderZ()
//
//	Cylinder with its principal axis along the Z-axis.
//
void QzMatrix3x3::TensorCylinderZ(float mass, float height, float radius)
{
	MakeZero();

	m_Matrix.c3x3[0][0] = ((mass * height * height) / 12.0f) + ((mass * radius * radius) / 4.0f);
	m_Matrix.c3x3[1][1] = ((mass * height * height) / 12.0f) + ((mass * radius * radius) / 4.0f);
	m_Matrix.c3x3[2][2] = ((mass * radius * radius) / 2.0f);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TensorSphere()
//
void QzMatrix3x3::TensorSphere(float mass, float radius)
{
	MakeZero();

	float t = (2.0f / 3.0f) * mass * radius * radius;

	m_Matrix.c3x3[0][0] = t;
	m_Matrix.c3x3[1][1] = t;
	m_Matrix.c3x3[2][2] = t;
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotateX()
//
void QzMatrix3x3::PreRotateX(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	QzMatrix3x3_t temp = m_Matrix;

	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[1][i] = (temp.c3x3[1][i] * c) + (temp.c3x3[2][i] * s);
		m_Matrix.c3x3[2][i] = (temp.c3x3[2][i] * c) - (temp.c3x3[1][i] * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotateX()
//
void QzMatrix3x3::PostRotateX(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	for (U32 i = 0; i < 3; ++i) {
		float t = m_Matrix.c3x3[i][1];
		m_Matrix.c3x3[i][1] = (t * c) - (m_Matrix.c3x3[i][2] * s);
		m_Matrix.c3x3[i][2] = (t * s) + (m_Matrix.c3x3[i][2] * c);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotateY()
//
void QzMatrix3x3::PreRotateY(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	QzMatrix3x3_t temp = m_Matrix;

	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[0][i] = (temp.c3x3[0][i] * c) - (temp.c3x3[2][i] * s);
		m_Matrix.c3x3[2][i] = (temp.c3x3[2][i] * c) + (temp.c3x3[0][i] * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotateY()
//
void QzMatrix3x3::PostRotateY(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	for (U32 i = 0; i < 3; ++i) {
		float t = m_Matrix.c3x3[i][0];
		m_Matrix.c3x3[i][0] = (m_Matrix.c3x3[i][2] * s) + (t * c);
		m_Matrix.c3x3[i][2] = (m_Matrix.c3x3[i][2] * c) - (t * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotateZ()
//
void QzMatrix3x3::PreRotateZ(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	QzMatrix3x3_t temp = m_Matrix;

	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[0][i] = (temp.c3x3[0][i] * c) + (temp.c3x3[1][i] * s);
		m_Matrix.c3x3[1][i] = (temp.c3x3[1][i] * c) - (temp.c3x3[0][i] * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotateZ()
//
void QzMatrix3x3::PostRotateZ(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	for (U32 i = 0; i < 3; ++i) {
		float t = m_Matrix.c3x3[i][0];
		m_Matrix.c3x3[i][0] = (t * c) - (m_Matrix.c3x3[i][1] * s);
		m_Matrix.c3x3[i][1] = (t * s) + (m_Matrix.c3x3[i][1] * c);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotate()
//
void QzMatrix3x3::PreRotate(QzQuat &quat)
{
	QzMatrix3x3 temp;
	quat.ConvertToMatrix(temp);
	PreMultiply(temp);
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotate()
//
void QzMatrix3x3::PostRotate(QzQuat &quat)
{
	QzMatrix3x3 temp;
	quat.ConvertToMatrix(temp);
	PostMultiply(temp);
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreScale()
//
void QzMatrix3x3::PreScale(float x, float y, float z)
{
	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[0][i] *= x;
		m_Matrix.c3x3[1][i] *= y;
		m_Matrix.c3x3[2][i] *= z;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreScale()
//
void QzMatrix3x3::PreScale(float x)
{
	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[0][i] *= x;
		m_Matrix.c3x3[1][i] *= x;
		m_Matrix.c3x3[2][i] *= x;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostScale()
//
void QzMatrix3x3::PostScale(float x, float y, float z)
{
	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[i][0] *= x;
		m_Matrix.c3x3[i][1] *= y;
		m_Matrix.c3x3[i][2] *= z;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostScale()
//
void QzMatrix3x3::PostScale(float x)
{
	for (U32 i = 0; i < 3; ++i) {
		m_Matrix.c3x3[i][0] *= x;
		m_Matrix.c3x3[i][1] *= x;
		m_Matrix.c3x3[i][2] *= x;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreTripleRotate()
//
void QzMatrix3x3::PreTripleRotate(const QzVector rotate)
{
	PreRotateY(rotate.m_Z);
	PreRotateX(rotate.m_X);
	PreRotateY(rotate.m_Y);
}


/////////////////////////////////////////////////////////////////////////////
//
//	ComputeInverseMatrix()
//
//	Find the inverse of the matrix by first computing the adjoint of the
//	original matrix, then dividing by the adjoint.  This is only processing
//	the upper-left 3x3 rotation matrix.  The translation part of the matrix
//	is extracted separately.
//
//	Obviously, this is not a generalized inverse.  It only works correctly on
//	homogeneous transforms, where the last row is the translation and the
//	last column is filled with zeroes.
//
void QzMatrix3x3::ComputeInverseMatrix(QzMatrix3x3 &src)
{
	float *s = src.m_Matrix.cFlat;
	float *d = m_Matrix.cFlat;

	// Compute the adjoint of the matrix.
	d[0] = (s[4] * s[8]) - (s[5] * s[7]);
	d[1] = (s[2] * s[7]) - (s[1] * s[8]);
	d[2] = (s[1] * s[5]) - (s[2] * s[4]);
	d[3] = (s[5] * s[6]) - (s[3] * s[8]);
	d[4] = (s[0] * s[8]) - (s[2] * s[6]);
	d[5] = (s[2] * s[3]) - (s[0] * s[5]);
	d[6] = (s[3] * s[7]) - (s[4] * s[6]);
	d[7] = (s[1] * s[6]) - (s[0] * s[7]);
	d[8] = (s[0] * s[4]) - (s[1] * s[3]);

	// Compute the determinant.  The determinant is derived from more complex
	// values.  But we just derived those values above, so we can reuse them
	// to directly extract the determinant.
	float determinant = (s[0] * d[0]) + (s[1] * d[3]) + (s[2] * d[6]);

	// Divide the adjoint by the determinant to compute the inverse rotation
	// matrix.  Assuming that we're always given a homogeneous transform,
	// the determinant should never be zero.  But always guard against
	// invalid divisions.
	if (false == QzFloatIsZero20(determinant)) {
		determinant = 1.0f / determinant;
		d[0] *= determinant;
		d[1] *= determinant;
		d[2] *= determinant;
		d[3] *= determinant;
		d[4] *= determinant;
		d[5] *= determinant;
		d[6] *= determinant;
		d[7] *= determinant;
		d[8] *= determinant;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ComputeNormalMatrix()
//
//	This is almost identical to computing the inverse of the matrix.  The
//	one difference is that we need to invert the transpose of the matrix.
//	So this code will zero out the transform portion of the matrix, and it
//	will normalize the rotation vectors so that unit vectors retain unit
//	length when rotated.
//
void QzMatrix3x3::ComputeNormalMatrix(QzMatrix3x3 &src)
{
	float *s = src.m_Matrix.cFlat;
	float *d = m_Matrix.cFlat;

	// Compute the adjoint of the transpose matrix.
	d[0] = (s[4] * s[8]) - (s[5] * s[7]);
	d[1] = (s[5] * s[6]) - (s[3] * s[8]);
	d[2] = (s[3] * s[7]) - (s[4] * s[6]);
	d[3] = (s[2] * s[7]) - (s[1] * s[8]);
	d[4] = (s[0] * s[8]) - (s[2] * s[6]);
	d[5] = (s[1] * s[6]) - (s[0] * s[7]);
	d[6] = (s[1] * s[5]) - (s[2] * s[4]);
	d[7] = (s[2] * s[3]) - (s[0] * s[5]);
	d[8] = (s[0] * s[4]) - (s[1] * s[3]);

	// Compute the determinant.  The determinant is derived from more complex
	// values.  But we just derived those values above, so we can reuse them
	// to directly extract the determinant.
	float determinant = (s[0] * d[0]) + (s[1] * d[2]) + (s[2] * d[2]);

	// Divide the adjoint by the determinant to compute the inverse rotation
	// matrix.  Assuming that we're always given a homogeneous transform,
	// the determinant should never be zero.  But always guard against
	// invalid divisions.
	if (false == QzFloatIsZero20(determinant)) {
		determinant = 1.0f / determinant;
		d[0] *= determinant;
		d[1] *= determinant;
		d[2] *= determinant;
		d[3] *= determinant;
		d[4] *= determinant;
		d[5] *= determinant;
		d[6] *= determinant;
		d[7] *= determinant;
		d[8] *= determinant;
	}

	// Now normalize the rotation vectors (since applying a rotation matrix
	// is effectively applying three dot products, so each of the three
	// vectors should be of unit length).
	//
	// Note that we're using QzEqualFloats10() here, since it's comparing the
	// square of the length.  Values close to 1.0 will be further from 1.0
	// before squaring, so we don't need as much precision on the comparison.
	//

	float len;

	len = (d[0] * d[0]) + (d[3] * d[3]) + (d[6] * d[6]);
	if (false == QzEqualFloats10(1.0f, len)) {
		len = sqrtf(len);
		d[0] /= len;
		d[3] /= len;
		d[6] /= len;
	}

	len = (d[1] * d[1]) + (d[4] * d[4]) + (d[7] * d[7]);
	if (false == QzEqualFloats10(1.0f, len)) {
		len = sqrtf(len);
		d[1] /= len;
		d[4] /= len;
		d[7] /= len;
	}

	len = (d[2] * d[2]) + (d[5] * d[5]) + (d[8] * d[8]);
	if (false == QzEqualFloats10(1.0f, len)) {
		len = sqrtf(len);
		d[2] /= len;
		d[5] /= len;
		d[8] /= len;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Multiply()
//
void QzMatrix3x3::Multiply(const QzMatrix3x3 &first, const QzMatrix3x3 &second)
{
	const QzMatrix3x3_t &m1 = first.m_Matrix;
	const QzMatrix3x3_t &m2 = second.m_Matrix;

	for (U32 row = 0; row < 3; ++row) {
		for (U32 col = 0; col < 3; ++col) {
			m_Matrix.c3x3[row][col] = m1.c3x3[row][0] * m2.c3x3[0][col]
									+ m1.c3x3[row][1] * m2.c3x3[1][col]
									+ m1.c3x3[row][2] * m2.c3x3[2][col];
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreMultiply()
//
void QzMatrix3x3::PreMultiply(const QzMatrix3x3 &first)
{
	const QzMatrix3x3_t &m1 = first.m_Matrix;
	const QzMatrix3x3_t  m2 = m_Matrix;

	for (U32 row = 0; row < 3; ++row) {
		for (U32 col = 0; col < 3; ++col) {
			m_Matrix.c3x3[row][col] = m1.c3x3[row][0] * m2.c3x3[0][col]
									+ m1.c3x3[row][1] * m2.c3x3[1][col]
									+ m1.c3x3[row][2] * m2.c3x3[2][col];
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostMultiply()
//
void QzMatrix3x3::PostMultiply(const QzMatrix3x3 &second)
{
	const QzMatrix3x3_t  m1 = m_Matrix;
	const QzMatrix3x3_t &m2 = second.m_Matrix;

	for (U32 row = 0; row < 3; ++row) {
		for (U32 col = 0; col < 3; ++col) {
			m_Matrix.c3x3[row][col] = m1.c3x3[row][0] * m2.c3x3[0][col]
									+ m1.c3x3[row][1] * m2.c3x3[1][col]
									+ m1.c3x3[row][2] * m2.c3x3[2][col];
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	TransformVector()
//
void QzMatrix3x3::TransformVector(const QzVector vec, QzVector &result)
{
	QzMatrix3x3_t &m = m_Matrix;

	result.m_X = (vec.m_X * m.c3x3[0][0]) + (vec.m_Y * m.c3x3[1][0]) + (vec.m_Z * m.c3x3[2][0]);
	result.m_Y = (vec.m_X * m.c3x3[0][1]) + (vec.m_Y * m.c3x3[1][1]) + (vec.m_Z * m.c3x3[2][1]);
	result.m_Z = (vec.m_X * m.c3x3[0][2]) + (vec.m_Y * m.c3x3[1][2]) + (vec.m_Z * m.c3x3[2][2]);
}


/////////////////////////////////////////////////////////////////////////////
//
//	TransformVector()
//
void QzMatrix3x3::TransformVector(const QzVector_t vec, QzVector_t &result)
{
	QzMatrix3x3_t &m = m_Matrix;

	result.X = (vec.X * m.c3x3[0][0]) + (vec.Y * m.c3x3[1][0]) + (vec.Z * m.c3x3[2][0]);
	result.Y = (vec.X * m.c3x3[0][1]) + (vec.Y * m.c3x3[1][1]) + (vec.Z * m.c3x3[2][1]);
	result.Z = (vec.X * m.c3x3[0][2]) + (vec.Y * m.c3x3[1][2]) + (vec.Z * m.c3x3[2][2]);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Transpose()
//
void QzMatrix3x3::Transpose(void)
{
	float *p = m_Matrix.cFlat;

	Swap(p[ 1], p[ 3]);
	Swap(p[ 2], p[ 6]);
	Swap(p[ 5], p[ 7]);
}


/////////////////////////////////////////////////////////////////////////////
//
//	LeftHandedToRightHanded()
//
void QzMatrix3x3::LeftHandedToRightHanded(const QzMatrix3x3 &src)
{
	m_Matrix.c3x3[0][0] =  src.m_Matrix.c3x3[0][0];
	m_Matrix.c3x3[0][1] =  src.m_Matrix.c3x3[0][1];
	m_Matrix.c3x3[0][2] = -src.m_Matrix.c3x3[0][2];
	m_Matrix.c3x3[1][0] =  src.m_Matrix.c3x3[1][0];
	m_Matrix.c3x3[1][1] =  src.m_Matrix.c3x3[1][1];
	m_Matrix.c3x3[1][2] = -src.m_Matrix.c3x3[1][2];
	m_Matrix.c3x3[2][0] = -src.m_Matrix.c3x3[2][0];
	m_Matrix.c3x3[2][1] = -src.m_Matrix.c3x3[2][1];
	m_Matrix.c3x3[2][2] =  src.m_Matrix.c3x3[2][2];
}


/////////////////////////////////////////////////////////////////////////////
//
//	LogMatrix()
//
void QzMatrix3x3::LogMatrix(void)
{
	QzMatrix3x3_t &m = m_Matrix;

	UtfFormat fmt;
	fmt.AddFloat(m.c3x3[0][0]);
	fmt.AddFloat(m.c3x3[0][1]);
	fmt.AddFloat(m.c3x3[0][2]);
	LogMessage("Matrix: %1w8.5; %2w8.5; %3w8.5;", fmt);

	fmt.Reset();
	fmt.AddFloat(m.c3x3[1][0]);
	fmt.AddFloat(m.c3x3[1][1]);
	fmt.AddFloat(m.c3x3[1][2]);
	LogMessage("        %1w8.5; %2w8.5; %3w8.5;", fmt);

	fmt.Reset();
	fmt.AddFloat(m.c3x3[2][0]);
	fmt.AddFloat(m.c3x3[2][1]);
	fmt.AddFloat(m.c3x3[2][2]);
	LogMessage("        %1w8.5; %2w8.5; %3w8.5;", fmt);
}


/////////////////////////////////////////////////////////////////////////////
//
//	PrintMatrix()
//
void QzMatrix3x3::PrintMatrix(void)
{
	QzMatrix3x3_t &m = m_Matrix;

	printf("Matrix: %8.5f %8.5f %8.5f\n", m.c3x3[0][0], m.c3x3[0][1], m.c3x3[0][2]);
	printf("        %8.5f %8.5f %8.5f\n", m.c3x3[1][0], m.c3x3[1][1], m.c3x3[1][2]);
	printf("        %8.5f %8.5f %8.5f\n", m.c3x3[2][0], m.c3x3[2][1], m.c3x3[2][2]);
	printf("\n");
}


