/////////////////////////////////////////////////////////////////////////////
//
//	File: QzManager.cpp
//
//	$Header: /Projects/Qz/QzManager.cpp  3  2009/9/14 5:44:08p  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzClient.h"
#include "QzKeyMapper.h"
#include "QzManager.h"
#include "QzRenderer.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


QzInterface* g_pQzInterface = NULL;


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzManager::QzManager(void)
	:	m_pClient(NULL),
		m_pKeyMapper(NULL),
		m_pRenderer(NULL),
		m_FrameCount(0),
		m_FrameCountLatch(0),
		m_FrameCountTick(0),
		m_TimestampPrevious(0),
		m_TimestampCurrent(0),
		m_CpuUsage(0.0f),
		m_ShowFPS(true),
		m_IsWindowDirty(true),
		m_IsWindowActive(false),
		m_IsMouseExclusive(false)
{
	m_pClient    = InstanceQzClient();
	m_pKeyMapper = new QzKeyMapper;
	m_pRenderer  = new QzRenderer;

	U32 currentTime = QzGetMilliseconds();

	m_TimestampPrevious = currentTime;
	m_TimestampCurrent  = currentTime;
	m_FrameCountTick    = currentTime;

	g_pQzInterface = this;
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzManager::~QzManager(void)
{
	g_pQzInterface = NULL;

	SafeDelete(m_pClient);
	SafeDelete(m_pKeyMapper);
	SafeDelete(m_pRenderer);
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetFrameDuration()
//
//	Returns the number of milliseconds ellapsed since the start of the
//	previous frame.
//
//	This value can be zero, especially when starting up, or when restored
//	after having been minimized.
//
U32 QzManager::GetFrameDuration(void)
{
	return (m_TimestampCurrent - m_TimestampPrevious);
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetKeyState()
//
//	Returns the state of the specified key.  This is relative to the start
//	of the current frame, based on the known state of the key.
//
//	Given value should use the QzKey_... values.
//
U32 QzManager::GetKeyState(U32 key)
{
	return m_pKeyMapper->GetKeyState(key);
}


/////////////////////////////////////////////////////////////////////////////
//
//	IsMouseExclusive()
//
bool QzManager::IsMouseExclusive(void)
{
	return m_IsMouseExclusive;
}


/////////////////////////////////////////////////////////////////////////////
//
//	RequestAppTermination()
//
void QzManager::RequestAppTermination(void)
{
	QzTerminateApp();
}


/////////////////////////////////////////////////////////////////////////////
//
//	SetMouseExclusive()
//
//	Attempts to take over exclusive control of the mouse from all other apps.
//	This should be enabled when in FPS mode, and disabled for regular windowed
//	mode (when in the editor, accessing menus, game inventory screen, etc.).
//
void QzManager::SetMouseExclusive(bool exclusive)
{
	if (m_IsMouseExclusive && !exclusive) {
		m_IsMouseExclusive = false;
		QzLockMouse(false);
	}
	else if (!m_IsMouseExclusive && exclusive) {
		m_IsMouseExclusive = true;
		QzLockMouse(true);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ClearDirtyFlag()
//
void QzManager::ClearDirtyFlag(void)
{
	m_IsWindowDirty = false;
}


/////////////////////////////////////////////////////////////////////////////
//
//	SetDirtyFlag()
//
void QzManager::SetDirtyFlag(void)
{
	m_IsWindowDirty = true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Initialize()
//
bool QzManager::Initialize(void)
{
	// Abort if we can't find the font file.
	if (false == m_pRenderer->LoadFonts(reinterpret_cast<const Utf08_t*>("GuiFont.dat"))) {
		QzShowSystemDialogBox(reinterpret_cast<const Utf08_t*>("Cannot open GuiFont.dat"), reinterpret_cast<const Utf08_t*>("Error"), true);
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	CreateRenderer()
//
bool QzManager::CreateRenderer(void)
{
//	m_pRenderer->LogExtensions();
	m_pRenderer->Create();

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DestroyRenderer()
//
bool QzManager::DestroyRenderer(void)
{
	m_pRenderer->Destroy();

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DeviceInputClear()
//
void QzManager::DeviceInputClear(void)
{
	m_pKeyMapper->ClearEvents();
}


/////////////////////////////////////////////////////////////////////////////
//
//	DeviceInputBegin()
//
//	This gets called whenever the window is activated.  It needs to reset
//	the known state of all keys and prepare to receive input.
//
void QzManager::DeviceInputBegin(void)
{
	// Reset the input so we don't carry any key state info over from
	// whenever the app was last active.
	m_pKeyMapper->Reset();

	m_pKeyMapper->DisableWindowsKey();
	m_pKeyMapper->DisableStickyKeys();

	// Call this now to reset the sampling data.
	QzGetProcessorUsage();
}


/////////////////////////////////////////////////////////////////////////////
//
//	DeviceInputEnd()
//
//	This is called whenever the window is deactived or otherwise looses the
//	input focus.  Need to undo any system-level changes or input filtering
//	that was put in place by DeviceInputBegin();
//
void QzManager::DeviceInputEnd(void)
{
	m_pKeyMapper->EnableWindowsKey();
	m_pKeyMapper->EnableStickyKeys();
}


/////////////////////////////////////////////////////////////////////////////
//
//	IsWindowDirty()
//
bool QzManager::IsWindowDirty(void)
{
	return m_IsWindowDirty;
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetAppTitle()
//
void QzManager::GetAppTitle(Utf08_t buffer[], U32 bufferSize)
{
	m_pClient->GetAppTitle(buffer, bufferSize);
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetMaxFrameDelay()
//
//	Reports the maximum number of milliseconds that should ellapse between
//	redrawing the window if the window is not dirty/invalidated.  When the
//	window is marked as dirty, it will be redrawn as soon as the app gets
//	chance to redraw.
//
U32 QzManager::GetMaxFrameDelay(void)
{
	U32 duration = m_pClient->GetMaxFrameDelay();

	return ClampRange(10, duration, 1000);
}


/////////////////////////////////////////////////////////////////////////////
//
//	SetResolution()
//
void QzManager::SetResolution(U32 width, U32 height)
{
	m_pRenderer->SetResolution(width, height);
}


/////////////////////////////////////////////////////////////////////////////
//
//	KeyPress()
//
//	The <isSystem> flag is used on Windows when pressing F10, Alt, or any
//	Alt-key combination.  If we don't care about Window's accelerator logic,
//	we can ignore the <isSystem> flag and process all key presses the same.
//
void QzManager::KeyPress(U32 keyFlags, U32 key, bool isDown, bool isSystem)
{
	// Record the key press.  This will track all events and update the state
	// for any keys that were pressed or released.
	m_pKeyMapper->KeyPress(key, isDown);

	if (isDown && !isSystem) {
		if (m_pKeyMapper->TranslateKey(keyFlags, key)) {
			m_pClient->KeyPress(keyFlags, key);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	KeyChar()
//
void QzManager::KeyChar(Utf32_t keyChar)
{
	m_pClient->KeyChar(keyChar);
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseMove()
//
void QzManager::MouseMove(S32 x, S32 y)
{
	m_pClient->MouseMove(x, y);
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseDelta()
//
void QzManager::MouseDelta(S32 dx, S32 dy)
{
	m_pClient->MouseDelta(dx, dy);
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseClick()
//
void QzManager::MouseClick(QzMouseButton_t button, QzMouseClick_t click, U32 flags, S32 x, S32 y)
{
	m_pClient->MouseClick(button, click, flags, x, y);
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseWheel()
//
void QzManager::MouseWheel(S32 delta)
{
	m_pClient->MouseWheel(delta);
}


/////////////////////////////////////////////////////////////////////////////
//
//	WindowActivated()
//
//	Called when the app gains or loses focus.  Need to unlock the mouse when
//	in exclusive mode so other apps will be able to receive input.
//
void QzManager::WindowActivated(bool activated)
{
	UtfFormat fmt;
	fmt.AddString(activated ? "true" : "false");
	fmt.AddString(m_IsWindowActive ? "true" : "false");
	LogMessage("Activated: %1;, was %2;", fmt);

	if (m_IsWindowActive != activated) {
		m_IsWindowActive = activated;
		if (m_IsWindowActive) {
			DeviceInputBegin();

			// After having been minimized, need to fix the timestamps used
			// for computing the CPU usage and frame rate.
			m_FrameCount        = 0;
			m_TimestampPrevious = QzGetMilliseconds();
			m_TimestampCurrent  = m_TimestampPrevious;
			m_FrameCountTick    = m_TimestampPrevious;

			// If the mouse should be locked, relock it when we regain focus.
			if (m_IsMouseExclusive) {
				QzLockMouse(true);
			}
		}
		else {
			DeviceInputEnd();

			// Make certain the mouse is unlocked when we lose focus.
			QzLockMouse(false);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Render()
//
bool QzManager::Render(void)
{
	// Avoid attempting to draw before the renderer has been created.
	// This can happen when the app first starts.
	if (false == m_pRenderer->IsCreated()) {
		return false;
	}

	m_IsWindowDirty = false;

	U32 currentTime = QzGetMilliseconds();

	// Track the render time for the current and previous frame.  This will
	// be needed in real-time mode to properly update animations.
	m_TimestampPrevious = m_TimestampCurrent;
	m_TimestampCurrent  = currentTime;

	++m_FrameCount;

	// Once per second, update the frame rate and CPU usage meters.
	if ((currentTime - m_FrameCountTick) >= 1000) {
		m_FrameCountTick += 1000;
		m_FrameCountLatch = m_FrameCount;
		m_FrameCount      = 0;

		m_CpuUsage = 100.0f * QzGetProcessorUsage();
	}

	m_pClient->Render(m_pRenderer);

	if (m_ShowFPS) {
		UtfFormat fmt;
		Utf08_t strg[128];

		fmt.Reset();
		fmt.AddInt(m_FrameCountLatch);
		fmt.Generate(strg, ArraySize(strg), reinterpret_cast<const Utf08_t*>("FPS %1w4;"));
		m_pRenderer->FontRender(QzFont_MonospaceHi, m_pRenderer->GetWidth() - 100, 10, strg);

		fmt.Reset();
		fmt.AddFloat(m_CpuUsage);
		fmt.Generate(strg, ArraySize(strg), reinterpret_cast<const Utf08_t*>("CPU %1w3.0;%%"));
		m_pRenderer->FontRender(QzFont_MonospaceHi, m_pRenderer->GetWidth() - 100, 30, strg);
	}

	m_pRenderer->RenderFinish();

	// Discard any remaining events.  They will no longer be needed for the
	// current frame.
	DeviceInputClear();

	return true;
}




