/////////////////////////////////////////////////////////////////////////////
//
//	File: QzMainWin.cpp
//
//	$Header: /Projects/Qz/QzMainWin.cpp  5  2009/9/15 10:57:32a  Lee $
//
//
//	This class is only used when building on Windows.  It implements the
//	front end for dealing with mouse and keyboard input, and managing the
//	window in which we will be rendering.
//
//	All events from this class are converted to a platform-independent format
//	and passed to QzManager, which allows QzManager to remain completely
//	ignorant of the platform upon which it is running.
//
/////////////////////////////////////////////////////////////////////////////


#include <windows.h>
#include <signal.h>
#include "QzCommon.h"
#include "QzLogger.h"
#include "QzMainWin.h"
#include "QzManager.h"
#include "QzRenderer.h"
#include "UtfData.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


// A special "wiggle" routine is required to enable/disable v-sync.
typedef BOOL (__stdcall *wlgSwapInterval_t)(int enable);
wlgSwapInterval_t g_SwapInt = NULL;

// This is instanced in QzSystemWin.cpp.
extern HWND g_hWindow;

// This is the window class.  It needs a global pointer so the WindowProc()
// callback handler can forward messages to this object.
static QzMainWin *g_pQzMain = NULL;

// We need to make this a global object so the renderer can reference it
// when calling SwapBuffers().
HDC g_hDC = NULL;

#define		c_WindowClass	L"QzWinMain"

#define		c_hTimer		123


/////////////////////////////////////////////////////////////////////////////
//
//	QzAssertHandler()
//
//	Custom assertion handler.  Deals with assertions without throwing up a
//	dialog, which doesn't necessary work right when running full-screen.
//
void QzAssertHandler(char message[], U32 lineNum, char file[])
{
	char *pattern = "Assert: line %1;, file %2;\n\n%3;\n";

	UtfFormat fmt;
	fmt.AddInt(lineNum);
	fmt.AddString(file);
	fmt.AddString(message);

	// Log the message.
	LogMessage(pattern, fmt);

	// Close the log file so we don't lose the assertion message.
	SafeDelete(g_pLog);

	// Use DebugBreak to launch the debugger ...
//	DebugBreak();

	// ... or just kill the app and drop back to the desktop.
	raise(SIGABRT);
}


/////////////////////////////////////////////////////////////////////////////
//
//	WindowProc()
//
//	This is the callback function presented to Windows.  It redirects the
//	callbacks into the main object for processing.
//
LRESULT WINAPI WindowProc(HWND hWindow, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (NULL != g_pQzMain) {
		return g_pQzMain->MessageHandler(hWindow, message, wParam, lParam);
	}

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzMainWin::QzMainWin(void)
	:	m_pManager(m_pManager),
		m_hOpenglRC(NULL),
		m_WindowActive(false),
		m_WindowMinimized(false),
		m_WindowWidth(640),
		m_WindowHeight(480),
		m_LeftDown(false),
		m_RightDown(false),
		m_HasCapture(false),
		m_FrameCount(0),
		m_FullScreen(false)
{
	m_pManager = new QzManager;
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzMainWin::~QzMainWin(void)
{
	SafeDelete(m_pManager);
}


/////////////////////////////////////////////////////////////////////////////
//
//	InitPixelFormat()
//
bool QzMainWin::InitPixelFormat(HDC hDC)
{
	PIXELFORMATDESCRIPTOR format;

	SafeZeroVar(format);

	format.nSize		= sizeof(PIXELFORMATDESCRIPTOR);
	format.nVersion		= 1;
	format.dwFlags		= PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	format.dwLayerMask	= PFD_MAIN_PLANE;
	format.iPixelType	= PFD_TYPE_RGBA;
	format.cColorBits	= 24;
	format.cDepthBits	= 0;
	format.cAccumBits	= 0;
	format.cStencilBits	= 0;

	int formatIndex = ChoosePixelFormat(hDC, &format);

	if (0 == formatIndex) {
		MessageBox(NULL, L"ChoosePixelFormat failed", L"Error", MB_OK);
		return false;
	}

	if (SetPixelFormat(hDC, formatIndex, &format) == FALSE) {
		MessageBox(NULL, L"SetPixelFormat failed", L"Error", MB_OK);
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ResizeWindow()
//
void QzMainWin::ResizeWindow(long width, long height)
{
	m_WindowWidth  = width;
	m_WindowHeight = height;

	m_pManager->SetResolution(width, height);
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleResized()
//
//	The window has been resized.  Find out the new resolution and update the
//	renderer to render at this resolution.
//
//	Need to explicitly pass in hWindow instead of using g_hWindow, since
//	g_hWindow will not yet have been initialized the first time this method
//	is called (in response to a WM_SIZE message).
//
void QzMainWin::HandleResized(HWND hWindow)
{
	RECT rect;

	GetClientRect(hWindow, &rect);

	long width  = rect.right - rect.left;
	long height = rect.bottom - rect.top;

	if ((width > 0) && (height > 0)) {
		m_WindowWidth  = width;
		m_WindowHeight = height;
		ResizeWindow(width, height);
	}

	GetWindowRect(hWindow, &m_WindowRect);
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdateMouseCapture()
//
//	Captures or releases mouse input depending upon whether any mouse buttons
//	are down.  Once captured, mouse input will not be released until all of
//	the mouse buttons are released.
//
void QzMainWin::UpdateMouseCapture(void)
{
	if (m_HasCapture) {
		if ((false == m_LeftDown) && (false == m_RightDown)) {
			m_HasCapture = false;
			ReleaseCapture();
		}
	}
	else {
		if (m_LeftDown || m_RightDown) {
			m_HasCapture = true;
			SetCapture(g_hWindow);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleHighDefInput()
//
//	Processes WM_INPUT messages for the mouse.
//
void QzMainWin::HandleHighDefInput(LPARAM lParam)
{
	U08 buffer[1024];

	UINT size;
	GetRawInputData(HRAWINPUT(lParam), RID_INPUT, NULL, &size, sizeof(RAWINPUTHEADER));

	QzAssert(size <= sizeof(buffer));

	if (size != GetRawInputData((HRAWINPUT)lParam, RID_INPUT, buffer, &size, sizeof(RAWINPUTHEADER))) {
		QzAssertAlways("GetRawInputData did not return correct size");
	}

	RAWINPUT *pInput = reinterpret_cast<RAWINPUT*>(buffer);

	if (RIM_TYPEKEYBOARD == pInput->header.dwType) {
/*
		LogMessage("Kbd: make=%04x Flags:%04x Reserved:%04x ExtraInformation:%08x, msg=%04x VK=%04x",
			pInput->data.keyboard.MakeCode,
			pInput->data.keyboard.Flags,
			pInput->data.keyboard.Reserved,
			pInput->data.keyboard.ExtraInformation,
			pInput->data.keyboard.Message,
			pInput->data.keyboard.VKey);
*/
	}
	else if (RIM_TYPEMOUSE == pInput->header.dwType) {
/*
		LogMessage("Mouse: usFlags=%04x ulButtons=%04x usButtonFlags=%04x usButtonData=%04x ulRawButtons=%04x lLastX=%4d lLastY=%4d ulExtraInformation=%04x",
			pInput->data.mouse.usFlags,
			pInput->data.mouse.ulButtons,
			pInput->data.mouse.usButtonFlags,
			pInput->data.mouse.usButtonData,
			pInput->data.mouse.ulRawButtons,
			pInput->data.mouse.lLastX,
			pInput->data.mouse.lLastY,
			pInput->data.mouse.ulExtraInformation);
*/
		m_pManager->MouseDelta(pInput->data.mouse.lLastX, pInput->data.mouse.lLastY);

		if (RI_MOUSE_LEFT_BUTTON_DOWN & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseClick(QzMouseButton_Left, QzMouseClick_Down, 0, 0, 0);
		}

		if (RI_MOUSE_LEFT_BUTTON_UP & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseClick(QzMouseButton_Left, QzMouseClick_Up, 0, 0, 0);
		}

		if (RI_MOUSE_MIDDLE_BUTTON_DOWN & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseClick(QzMouseButton_Middle, QzMouseClick_Down, 0, 0, 0);
		}

		if (RI_MOUSE_MIDDLE_BUTTON_UP & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseClick(QzMouseButton_Middle, QzMouseClick_Up, 0, 0, 0);
		}

		if (RI_MOUSE_RIGHT_BUTTON_DOWN & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseClick(QzMouseButton_Right, QzMouseClick_Down, 0, 0, 0);
		}

		if (RI_MOUSE_RIGHT_BUTTON_UP & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseClick(QzMouseButton_Right, QzMouseClick_Up, 0, 0, 0);
		}

		if (RI_MOUSE_WHEEL & pInput->data.mouse.usButtonFlags) {
			m_pManager->MouseWheel(S32(S16(pInput->data.mouse.usButtonData)));
		}
	}

	// Keep the mouse centered in the middle of the window.  This will prevent
	// any clicks from activating background windows when running in windowed
	// mode.
	//
	// For effeciency, should only do this once per frame, instead of after
	// every single WM_INPUT message, but that would require a method to test
	// whether GuiPrivate has the mouse locked or not.  This should not be done
	// when in window mode.
	//
	SetCursorPos((m_WindowRect.left + m_WindowRect.right) / 2, (m_WindowRect.top + m_WindowRect.bottom) / 2);
}


// Convert the 16-bit x,y coords into properly sign-extended ints.
#define TransX S32(S16(LOWORD(lParam)))
#define TransY S32(S16(HIWORD(lParam)))


/////////////////////////////////////////////////////////////////////////////
//
//	WindowProc()
//
LRESULT QzMainWin::MessageHandler(HWND hWindow, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		// App has gained or lost focus.
		case WM_ACTIVATE:
			{
				bool activated = (WA_INACTIVE != LOWORD(wParam));
				bool minimized = (0 != HIWORD(wParam));
				if (activated && !minimized) {
					m_WindowActive = true;
					m_pManager->WindowActivated(true);

					if (m_WindowMinimized) {
						m_WindowMinimized = false;
						HandleResized(hWindow);
					}
				}
				else {
					if (minimized) {
						m_WindowMinimized = true;
					}

					m_WindowActive = false;
					m_pManager->WindowActivated(false);
				}
			}
			break;

		case WM_EXITSIZEMOVE:
			HandleResized(hWindow);
			break;

		case WM_MOVE:
			HandleResized(hWindow);
			break;

		case WM_SIZE:
			HandleResized(hWindow);
			break;

		// These messages should only be received while the game is in FPS
		// mode.  The rest of the time, when it is in windowed mode, it uses
		// the standard WM_MOUSE... messages.  GuiPrivate is responsible for
		// switching high-def input on and offset as it requires.
		case WM_INPUT:
			HandleHighDefInput(lParam);
			return 0;

		case WM_CHAR:
			m_pManager->KeyChar(Utf32_t(wParam));
			return 0;

		case WM_KEYDOWN:
			m_pManager->KeyPress(U32(lParam), U32(wParam), true, false);
			return 0;

		case WM_SYSKEYDOWN:
			m_pManager->KeyPress(U32(lParam), U32(wParam), true, true);
			return 0;

		case WM_SYSKEYUP:
			m_pManager->KeyPress(U32(lParam), U32(wParam), false, true);
			return 0;

		case WM_KEYUP:
			m_pManager->KeyPress(U32(lParam), U32(wParam), false, false);
			return 0;

		case WM_MOUSEWHEEL:
			m_pManager->MouseWheel(S16(HIWORD(wParam)) / 120);
			break;

		case WM_MOUSEMOVE:
			m_pManager->MouseMove(TransX, TransY);
			break;

		case WM_LBUTTONDOWN:
			m_LeftDown = true;
			UpdateMouseCapture();
			m_pManager->MouseClick(QzMouseButton_Left, QzMouseClick_Down, U32(wParam), TransX, TransY);
			break;

		case WM_LBUTTONUP:
			m_LeftDown = false;
			UpdateMouseCapture();
			m_pManager->MouseClick(QzMouseButton_Left, QzMouseClick_Up, U32(wParam), TransX, TransY);
			break;

		case WM_LBUTTONDBLCLK:
			m_LeftDown = true;
			UpdateMouseCapture();
			m_pManager->MouseClick(QzMouseButton_Left, QzMouseClick_Double, U32(wParam), TransX, TransY);
			break;

		case WM_RBUTTONDOWN:
			m_RightDown = true;
			UpdateMouseCapture();
			m_pManager->MouseClick(QzMouseButton_Right, QzMouseClick_Down, U32(wParam), TransX, TransY);
			break;

		case WM_RBUTTONUP:
			m_RightDown = false;
			UpdateMouseCapture();
			m_pManager->MouseClick(QzMouseButton_Right, QzMouseClick_Up, U32(wParam), TransX, TransY);
			break;

		case WM_RBUTTONDBLCLK:
			m_RightDown = true;
			UpdateMouseCapture();
			m_pManager->MouseClick(QzMouseButton_Right, QzMouseClick_Double, U32(wParam), TransX, TransY);
			break;

		case WM_TIMER:
			// A timer is used to periodically poll the window manager to
			// determine if the window needs to be redrawn.  If the window
			// is dirty (its internal state has changed), or a minimum amount
			// of time has passed, the window will be redrawn.
			//
			// However, the actual logic for this test is handled within
			// the message loop, so that the dirty and time tests can be
			// done every time a message is handled by the message loop.
			//
			if (c_hTimer == wParam) {
			}
			break;

		case WM_ERASEBKGND:
			return TRUE;

		case WM_PAINT:
			{
				PAINTSTRUCT ps;
				BeginPaint(hWindow, &ps);
				EndPaint(hWindow, &ps);

				// Only render when the window is active and has the input
				// focus, so we don't waste rendering resources for hidden
				// windows (which is rude to other windows).
				if (m_WindowActive) {
					m_pManager->Render();
					++m_FrameCount;
				}
			}
			break;

		case WM_CLOSE:
			m_pManager->DestroyRenderer();

			if (m_hOpenglRC) {
				wglDeleteContext(m_hOpenglRC);
				m_hOpenglRC = NULL;
			}

			if (g_hDC) {
				ReleaseDC(hWindow, g_hDC);
				g_hDC = NULL;
			}

			DestroyWindow(hWindow);
			break;

		case WM_DESTROY:
			if (NULL != m_hOpenglRC) {
				wglDeleteContext(m_hOpenglRC);
				m_hOpenglRC = NULL;
			}

			if (NULL != g_hDC) {
				ReleaseDC(hWindow, g_hDC);
				g_hDC = NULL;
			}

			PostQuitMessage(0);
			break;

		default:
			return DefWindowProc(hWindow, message, wParam, lParam);
	}

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	RegisterWindowClass()
//
ATOM QzMainWin::RegisterWindowClass(HINSTANCE hInstance)
{
	WNDCLASSEX windowClass;

	// OpenGL needs the CS_OWNDC flag set.

	windowClass.cbSize			= sizeof(WNDCLASSEX);
	windowClass.style			= CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	windowClass.lpfnWndProc		= WNDPROC(WindowProc);
	windowClass.cbClsExtra		= 0;
	windowClass.cbWndExtra		= 0;
	windowClass.hInstance		= hInstance;
	windowClass.hIcon			= LoadIcon(hInstance, LPCTSTR(IDI_APPLICATION));
	windowClass.hCursor			= LoadCursor(NULL, IDC_ARROW);
	windowClass.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);
	windowClass.lpszMenuName	= NULL;
	windowClass.lpszClassName	= c_WindowClass;
	windowClass.hIconSm			= NULL;

	return RegisterClassEx(&windowClass);
}


/////////////////////////////////////////////////////////////////////////////
//
//	InitInstance()
//
HWND QzMainWin::InitInstance(HINSTANCE hInstance, int showCommand, Utf08_t appTitle[])
{
	DWORD style = WS_OVERLAPPEDWINDOW;// | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;

	long cx = GetSystemMetrics(SM_CXFULLSCREEN);
	long cy = GetSystemMetrics(SM_CYFULLSCREEN);

	RECT rect;

	if (m_FullScreen) {
		HWND hDesktopWindow = GetDesktopWindow();
		HDC  hDesktopDC     = GetDC(hDesktopWindow);

		style       = WS_POPUP | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
		rect.left   = 0;
		rect.top    = 0;
		rect.right  = GetDeviceCaps(hDesktopDC, HORZRES);
		rect.bottom = GetDeviceCaps(hDesktopDC, VERTRES);

		ReleaseDC(hDesktopWindow, hDesktopDC);
	}
	else {
		rect.left   = 0;
		rect.top    = 0;
		rect.right  = m_WindowWidth;
		rect.bottom = m_WindowHeight;

		// Expand window dimensions so client rect is the requested size.
		AdjustWindowRect(&rect, style, FALSE);

		S32 offsetX = rect.right - rect.left;
		S32 offsetY = rect.bottom - rect.top;

		// Adjust horizontal and vertical offset so that the window will be
		// centered on the desktop.  Note that GetSystemMetrics() returns a
		// size value that excludes the taskbar.

		if (offsetX > cx) {
			offsetX = 0;
		}
		else {
			offsetX = (cx - offsetX) / 2;
		}

		if (offsetY > cy) {
			offsetY = 0;
		}
		else {
			offsetY = (cy - offsetY) / 2;
		}

		// Note the order of operation: we need to update the right/bottom
		// values first, otherwise they will be computed incorrectly.
		rect.right  = offsetX + (rect.right - rect.left);
		rect.left   = offsetX;
		rect.bottom = offsetY + (rect.bottom - rect.top);
		rect.top    = offsetY;
	}

	// Request the window title from the client.  This will be a UTF-8
	// string, so we need to convert it to UTF-16 for use by Win32 routines.
	Utf16_t title16[64];
	UtfConvert08to16(title16, ArraySize(title16), appTitle);

	HWND hWindow = CreateWindow(
					c_WindowClass,
					title16,
					style,
					rect.left,
					rect.top,
					rect.right - rect.left,
					rect.bottom - rect.top,
					NULL,
					NULL,
					hInstance,
					NULL);

	if (!hWindow) {
		return NULL;
	}

	ShowWindow(hWindow, showCommand);
	UpdateWindow(hWindow);

	return hWindow;
}


/////////////////////////////////////////////////////////////////////////////
//
//	EnableVSync()
//
//	Set the "swap interval" to FALSE to disable v-sync.  It may default to on
//	or off depending on system configuration.  Exercise caution when turning
//	it off, as higher-end graphics cards can potentially run at several
//	thousand FPS for simple rendering tasks (usually only in full-screen
//	mode), which tends to do bad things on systems with heat problems.
//
//	NOTE: Even if wglSwapIntervalEXT() exists, it may not work depending on
//	the firmware installed for the video card.  So even if we can set the
//	value to 1, the app may still render at very high FPS.
//
//	NOTE: Enabling v-sync in OpenGL can cause the driver to busy-wait for
//	the correct time to page swap, chewing up 100% of the CPU time on one
//	core.  Unless the app would otherwise render at hundreds or thousands of
//	FPS (and this is the best way to limit FPS), it is best to leave this
//	turned off.
//
void QzMainWin::EnableVSync(BOOL enabled)
{
	if (NULL == g_SwapInt) {
		// Locate the function pointer needed to control v-sync.
		g_SwapInt = (wlgSwapInterval_t)wglGetProcAddress("wglSwapIntervalEXT");
	}

	if (NULL != g_SwapInt) {
		g_SwapInt(enabled ? 1 : 0);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Init()
//
bool QzMainWin::Init(HINSTANCE hInstance, int showCommand)
{
	RECT      rect;
	UtfFormat fmt;

	if (false == m_pManager->Initialize()) {
		return false;
	}

	if (NULL == RegisterWindowClass(hInstance)) {
		return false;
	}

	Utf08_t appTitle[64];
	m_pManager->GetAppTitle(appTitle, ArraySize(appTitle));

	g_hWindow = InitInstance(hInstance, showCommand, appTitle);

	if (NULL == g_hWindow) {
		return false;
	}

	g_hDC = GetDC(g_hWindow);

	if (false == InitPixelFormat(g_hDC)) {
		return false;
	}

	m_hOpenglRC = wglCreateContext(g_hDC);

	if (NULL == m_hOpenglRC) {
		return false;
	}

	wglMakeCurrent(g_hDC, m_hOpenglRC);

	EnableVSync(FALSE);

	if (false == m_pManager->CreateRenderer()) {
		return false;
	}

	GetClientRect(g_hWindow, &rect);
	ResizeWindow(rect.right - rect.left, rect.bottom - rect.top);

	// Note that XP limits the minimum timer value to 10 milliseconds.
	// But hyperthreaded/multicore CPUs have a 16.7 millisecond resolution
	// on the timer, so the timer will arrive at most 66 times per second.
	SetTimer(g_hWindow, c_hTimer, 10, NULL);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Run()
//
int QzMainWin::Run(void)
{
	UtfFormat fmt;

	U32 t0, t1;

	MSG message;

	SafeZeroVar(message);

	t0 = QzGetMilliseconds();

	S32 result;

	U32 lastRedrawTime = t0;

#if 1

	//
	// This is a standard windows message loop.  It has the advantage of
	// using no CPU time except when there are messages to process, but the
	// short-coming that it is not the most stable way of sustaining a
	// consistent frame rate (e.g., when driving a game).
	//
	// It will only redraw the screen when the window state has changed
	// (e.g., the user clicked on a button), so very few frames will be
	// drawn per second, but this causes the app to idle at 0% CPU usage.
	// This makes the app considerate of all other apps that are running on
	// the system.
	//

	do {
		result = GetMessage(&message, NULL, 0, 0);

		if (-1 == result) {
			LogErrorMessage("GetMessage() error");
		}
		else if (result > 0) {
			TranslateMessage(&message);
			DispatchMessage(&message);
		}

		// After each message has been processed, check to see if the window
		// needs to be redrawn.  If the window is dirty, or a minimum amount
		// of time has passed (usually set to one second) since the last time
		// the window was redrawn, let's force a redraw.

		U32 timeStamp = QzGetMilliseconds();

		if (m_pManager->IsWindowDirty() ||
			((timeStamp - lastRedrawTime) >= m_pManager->GetMaxFrameDelay()))
		{
			m_pManager->ClearDirtyFlag();
			lastRedrawTime = timeStamp;
			InvalidateRect(g_hWindow, NULL, FALSE);
		}
	} while (0 != result);

#else

	//
	// This is a custom message loop that busy-waits whenever the app is
	// the active window.  This obviously chews up lots of CPU time if
	// nothing else is occurring, but gives better responsiveness to state
	// updates (e.g., consistent frame rate in games).
	//

	BOOL newMessage;
	do {
		// This message loop is used while the window is active.  It tries to
		// keep rendering running as fast as possible by processing all pending
		// messages at once, then rendering a new frame.
		if (m_WindowActive) {
			// Consume all messages currently sitting in the queue.
			do {
				newMessage = PeekMessage(&message, NULL, 0, 0, PM_NOREMOVE);

				if (newMessage && (WM_QUIT != message.message)) {
					if (GetMessage(&message, NULL, 0, 0)) {
						TranslateMessage(&message);
						DispatchMessage(&message);
					}
				}
			} while (newMessage && (WM_QUIT != message.message));

			if (m_pManager->IsWindowDirty()) {
				InvalidateRect(g_hWindow, NULL, FALSE);
			}
		}

		// The alternate loop is used when the window is inactive -- typically
		// minimized.  This will block and wait for new messages, processing
		// them as they arrive, without ever attempting to render anything.
		// Once the window is restored, this loop will break and revert to the
		// normal message-and-render loop above.
		else {
			while ((false == m_WindowActive) && GetMessage(&message, NULL, 0, 0)) {
				TranslateMessage(&message);
				DispatchMessage(&message);
			}
		}
	} while (WM_QUIT != message.message);

#endif

	t1 = QzGetMilliseconds();

	fmt.Reset();
	fmt.AddInt(m_FrameCount);
	fmt.AddInt(t1 - t0);
	fmt.AddFloat(float(m_FrameCount) * 1000.0f / float(t1 - t0));
	LogMessage("fps = %1; / %2; = %3;", fmt);

	return static_cast<int>(message.wParam);
}


/////////////////////////////////////////////////////////////////////////////
//
//	WinMain()
//
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE /*hPrevInstance*/, LPSTR /*cmdLine*/, int showCommand)
{
	QzSystemInit();
	UtfInitialize();

	QzLogger *pLog = new QzLogger("QzLog");
	g_pLog = pLog;

	// Only enable file logging if the memory logging option is not available.
	// This typically means the logger utility has not been started.
	if (!g_pLog->MemoryLoggingAvailable()) {
		g_pLog->Open(reinterpret_cast<const Utf08_t*>("trace.txt"));
	}

	// The main window needs to be a global pointer so it can be accessed
	// from WindowProc().
	g_pQzMain = new QzMainWin;

	int result = FALSE;

	if (g_pQzMain->Init(hInstance, showCommand)) {
		result = g_pQzMain->Run();
	}

	SafeDelete(g_pQzMain);

	g_pLog = NULL;

	SafeDelete(pLog);

	QzSystemUninit();

	_CrtDumpMemoryLeaks();

	return result;
}


