/////////////////////////////////////////////////////////////////////////////
//
//	File: QzKeyMapper.h
//
//	$Header: /TS/TsGui/QzKeyMapper.h  19  2009/9/8 12:43:34a  Lee $
//
//
//	This class is used to make system keyboard inputs to a standard set of
//	cross-platform symbols, record key events for later processing, and to
//	keep track of the state of each key, and whether any have been pressed or
//	released since the last update.
//
//	Since data processing occurs between world updates, the user may press
//	and release a key between render passes -- this is especially important
//	if the app loses focus or gets minimized.  Multiple key events can occur
//	before the next frame gets rendered.  This state logic allows the app to
//	test is a key was pressed or released, and whether it is still being held
//	down.
//
//	General usage assumes that ClearEvents() is being called after each frame
//	is rendered (or whenever the app is otherwise done processing the inputs
//	for the current frame).  The Reset() method is intended to clear all of
//	the key state information, which is a good idea any time the app has lost
//	keyboard focus, and therefore lost track of the state for each key.
//
/////////////////////////////////////////////////////////////////////////////


#pragma once


class QzKeyMapper
{
private:
	struct KeyState_t
	{
		U16 Flags;
		U16 PressCount;		// number of key-down events received (includes repeat counts) during last update
	};

	struct KeyEvent_t
	{
		U16 ID;				// which key, uses Key_... values, so this is index into m_KeyState
		U16 RepeatCount;
		U16 Down;			// FALSE = key-up, TRUE = key-down
	};

	KeyState_t	m_KeyState[QzKey_ArraySize];
	KeyEvent_t	m_KeyEvents[256];
	U32			m_KeyEventCount;

	// This is used for platform-specific data.  On Windows, it is used to
	// cache the sticky and windows keys.
	void*		m_pPrivate;

public:
	QzKeyMapper(void);
	~QzKeyMapper(void);

	void  KeyPress(U32 key, bool down);

	U32   GetKeyState(U32 key);
	U32   GetEvents(U32 events[], U32 maxEvents);

	bool  TranslateKey(U32 &flags, U32 &key);

	void  Reset(void);
	void  ClearEvents(void);

	void  SetFullScreen(bool fullScreen);

	void  DisableWindowsKey(void);
	void  EnableWindowsKey(void);
	void  DisableStickyKeys(void);
	void  EnableStickyKeys(void);
};


