/////////////////////////////////////////////////////////////////////////////
//
//	File: QzFontHandler.h
//
//	$Header: /Projects/Qz/QzFontHandler.h  2  2009/9/14 1:36:03p  Lee $
//
//
//	Supports rendering fonts extracted from a custom font file (named
//	"GuiFont.dat").  The fonts are custom made, avoiding any issues with
//	copyrights.  Also, the rendering technique is platform-independent,
//	which allows the exact same font to be used on all platforms -- this
//	avoids the problem where specific fonts don't exist on all platforms,
//	and worse, the problem that fonts with the same name don't look the
//	same on all platforms (many Microsoft fonts are different depending on
//	whether you have the English, European, or Asian versions).
//
//	All of the fonts from the file will be loaded, packed into a single
//	bitmap, and used in the same texture.  This allows more efficient
//	rendering when mixing multiple fonts (e.g., rendering text with some
//	words in bold or italic).
//
//	Note that this rendering logic assumes that the text has been fully
//	composed, so that any letters with diacritics (accent marks) have
//	been merged into single Unicode symbols.  Solitary diacritics will be
//	rendered as single characters.
//
/////////////////////////////////////////////////////////////////////////////


#pragma once


class QzByteStream;
class QzTexture;


struct QzFontChar_t
{
	Utf32_t Symbol;

	// The full size of the glyph, based on size of input TGA.
	U08 FullWidth;
	U08 FullHeight;

	// Width of symbol for spacing purposes.  Normally this is the same as
	// the FullWidth, but will be less when dealing with italic fonts.
	U08 SpacingWidth;

	// Left side of glyph for overlapping purposes.  Normally this is zero,
	// except for italic fonts.  This value pushes the glyph to the left
	// to fix the spacing, preventing italics from being spread out too much.
	U08 DeltaX;

	// The size of the char that actually has pixels in it.  Unneeded rows
	// and columns are left out of the pixel data to save space.  In some
	// cases, these may be zero, indicating that the character is empty
	// (this should only happen for spaces) -- do not render anything for
	// this char, just advance the character positioning accordingly.
	U08 ActiveWidth;
	U08 ActiveHeight;

	// Four sets of attachment points for diacritics: middle and right,
	// for both top and bottom.  The diacritic's attachment position must
	// be collocated with one of these positions.
	U08 HiMidX;
	U08 HiMidY;

	U08 HiRightX;
	U08 HiRightY;

	U08 LoMidX;
	U08 LoMidY;

	U08 LoRightX;
	U08 LoRightY;

	// Only valid for diacritics.  This is the position of the glyph that
	// needs to be aligned to the attachment point on the glyph to which it
	// is being attached.
	U08 AttachX;
	U08 AttachY;

	// Position of the first active pixel within the char.  This is used to
	// draw the active pixels at the correct location on the screen.  It is
	// also referenced to position diacritics just above or below the active
	// pixels so the diacritics float in roughly the correct place.
	U08 MinX;
	U08 MinY;

	// Position of the active pixels within the texture.
	U16 PosX;
	U16 PosY;
};


/////////////////////////////////////////////////////////////////////////////
//
struct QzFontInfo_t
{
	U32           SymbolCount;
	U32           MaxWidth;
	U32           MaxHeight;
	Utf08_t       Label[32];
	QzFontChar_t* pSymbols;
};


/////////////////////////////////////////////////////////////////////////////
//
class QzFontHandler
{
private:
	U08* m_pRawBuffer;
	U32  m_BufferSize;

	U32  m_ImageWidth;
	U32  m_ImageHeight;
	U32  m_BitDepth;		// only used for filling texture
	U08* m_pPixels;

	U32  m_GlyphCount;

	QzFontInfo_t	m_FontList[QzFont_ArraySize];

public:
	QzFontHandler(void);
	~QzFontHandler(void);

	bool LoadFontFile(const Utf08_t filename[]);
	void FillTexture(QzTexture *pTexture, U32 bitDepth);
	bool ScanFile(QzByteStream &stream, bool firstPass);
	bool ScanFont(QzByteStream &stream, bool firstPass);
	bool ScanSymbol(QzByteStream &stream, QzFontChar_t &info, bool firstPass);
	bool PlaceSymbolsInTexture(void);

	QzFontChar_t* LookUpChar(U32 fontID, Utf32_t c);

	U32  CharWidth(U32 fontID, Utf32_t c);
	U32  FontHeight(U32 fontID);
	U32  LimitCharsByWidth(U32 fontID, const Utf32_t strg[], U32 charCount, U32 maxWidth);
	void GetFontExtents(U32 fontID, const Utf32_t strg[], U32 charCount, U32 &width, U32 &height);
	void GetFontExtents(U32 fontID, const Utf08_t strg[], U32 byteCount, U32 &width, U32 &height);
	U32  ComputeQuadCount(U32 fontID, const Utf32_t strg[], U32 charCount);
	void DrawGlyph(U32 x, U32 y, U32 width, U32 height, float u0, float u1, float v0, float v1);
	U32  AssemblePixels(S32 x, S32 y, U32 color, U32 fontID, const Utf32_t strg[], U32 charCount);
	U32  AssemblePixelsCropped(S32 x, S32 y, const QzRect &crop, U32 color, U32 fontID, const Utf32_t strg[], U32 charCount);
	U32  AssemblePixelsFormatted(S32 &posX, S32 &posY, const QzRect &pos, U32 pColors[], U08 pFontID[], const Utf32_t strg[], U32 charCount);
};


