/////////////////////////////////////////////////////////////////////////////
//
//	File: QzFontHandler.cpp
//
//	$Header: /Projects/Qz/QzFontHandler.cpp  2  2009/9/14 1:36:08p  Lee $
//
//
//	Supports rendering fonts extracted from a custom font file (named
//	"GuiFont.dat").  The fonts are custom made, avoiding any issues with
//	copyrights.  Also, the rendering technique is platform-independent,
//	which allows the exact same font to be used on all platforms -- this
//	avoids the problem where specific fonts don't exist on all platforms,
//	and worse, the problem that fonts with the same name don't look the
//	same on all platforms (many Microsoft fonts are different depending on
//	whether you have the English, European, or Asian versions).
//
//	All of the fonts from the file will be loaded, packed into a single
//	bitmap, and used in the same texture.  This allows more efficient
//	rendering when mixing multiple fonts (e.g., rendering text with some
//	words in bold or italic).
//
//	Note that this rendering logic assumes that the text has been fully
//	composed, so that any letters with diacritics (accent marks) have
//	been merged into single Unicode symbols.  Solitary diacritics will be
//	rendered as single characters.
//
/////////////////////////////////////////////////////////////////////////////


#ifdef _WIN32
#include <windows.h>
#include <GL/gl.h>
#else
#include <OpenGL/OpenGL.h>
#include <AGL/agl.h>
#endif


#include "QzCommon.h"
#include "QzBuffer.h"
#include "QzByteStream.h"
#include "QzFourCC.h"
#include "QzFontHandler.h"
#include "QzRect.h"
#include "QzRectSorter.h"
#include "QzTexture.h"
#include "UtfData.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzFontHandler::QzFontHandler(void)
	:	m_pRawBuffer(NULL),
		m_BufferSize(0),
		m_ImageWidth(512),
		m_ImageHeight(512),
		m_BitDepth(8),
		m_pPixels(NULL)
{
	for (U32 i = 0; i < QzFont_ArraySize; ++i) {
		m_FontList[i].SymbolCount = 0;
		m_FontList[i].MaxWidth    = 1;
		m_FontList[i].MaxHeight   = 1;
		m_FontList[i].pSymbols    = NULL;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzFontHandler::~QzFontHandler(void)
{
	SafeDeleteArray(m_pPixels);

	for (U32 i = 0; i < QzFont_ArraySize; ++i) {
		SafeDeleteArray(m_FontList[i].pSymbols);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	LoadFontFile()
//
//	Reads in the font file and packs each symbol into one large texture.
//
bool QzFontHandler::LoadFontFile(const Utf08_t filename[])
{
	if (NULL == filename) {
		LogErrorMessage("QzFontHandler::LoadFontFile() failed, filename is NULL");
		return false;
	}

	// This method is only intended to be called once, so the pixel buffer
	// should always be NULL at this point.
	QzAssert(NULL == m_pPixels);

	m_pPixels = new U08[m_ImageWidth * m_ImageHeight];

	memset(m_pPixels, 0, m_ImageWidth * m_ImageHeight);


	//
	// Read the font file into a buffer so we can scan it.
	//

	QzBuffer fileData;
	if (false == fileData.ReadFromFile(filename)) {
		UtfFormat fmt;
		fmt.AddString(filename);
		LogErrorMessage("unable to open font file \"%1;\"", fmt);
		return false;
	}


	//
	// Compile the individual images into one large texture.
	//

	QzByteStream stream(fileData.GetAddress(), fileData.GetSize());

	// Perform an initial scan.  This will determine how many symbols there
	// are, and the size of each symbol.
	if (false == ScanFile(stream, true)) {
		return false;
	}

	// Now we can sort the symbols and determine where each will be stored
	// within the texture.  This can fail if the texture is too small.
	if (false == PlaceSymbolsInTexture()) {
		return false;
	}

	// Reset the byte stream so we can rescan the file.
	stream.StartStream(fileData.GetAddress(), fileData.GetSize());

	// Perform the second pass over the file.  This time we'll uncompress
	// each of the symbols into its alloted location within the texture.
	if (false == ScanFile(stream, false)) {
		return false;
	}

	bool success = true;

	// Make sure that all of the fonts are defined.  There needs to be at
	// least one symbol in each font, or LookUpChar() will violate memory
	// when queried for any symbols in that font.
	for (U32 i = 0; i < QzFont_ArraySize; ++i) {
		if (0 == m_FontList[i].SymbolCount) {
			UtfFormat fmt;
			fmt.AddInt(i);
			LogErrorMessage("font [%1;] does not contain any symbols", fmt);

			success = false;
		}
	}

	return success;
}


/////////////////////////////////////////////////////////////////////////////
//
//	FillTexture()
//
//	Given a texture buffer that already exists (but has presumably not been
//	initialized), we need to alloc memory within the buffer, then blit the
//	pixels into the texture so they can be fed down to the hardware.
//
void QzFontHandler::FillTexture(QzTexture *pTexture, U32 bitDepth)
{
	// For legacy support of fixed-function pipeline, we may need to store
	// the texture as a 32-bit image so we can be assured that the alpha
	// channel will work correctly.
	if (32 == bitDepth) {
		pTexture->Allocate(m_ImageWidth, m_ImageHeight, QzTexFormat_ARGB32);

		U08 *pPixels = pTexture->m_pBuffer;

		for (U32 i = 0; i < (m_ImageWidth * m_ImageHeight); ++i) {
			pPixels[(i*4)  ] = m_pPixels[i];
			pPixels[(i*4)+1] = m_pPixels[i];
			pPixels[(i*4)+2] = m_pPixels[i];
			pPixels[(i*4)+3] = m_pPixels[i];
		}
	}

	// Otherwise we should be using shaders, allowing the alpha-only texture
	// to be fed into the shader.  This allows explicit control over the font
	// rendering, and we reduce the memory needed to store the texture.
	else {
		pTexture->Allocate(m_ImageWidth, m_ImageHeight, QzTexFormat_Greyscale);

		U08 *pPixels = pTexture->m_pBuffer;

		memcpy(pPixels, m_pPixels, m_ImageWidth * m_ImageHeight);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ScanFile()
//
//	This performs a scan over the font file, parsing out each of the font
//	chunks stored therein.  This is coded to support two passes, the first
//	one gathering information and the second one decoding the compressed
//	symbols in each font.
//
bool QzFontHandler::ScanFile(QzByteStream &stream, bool firstPass)
{
	U32 cookie    = stream.ReadFourCC();
	U32 chunkSize = stream.ReadU32();

	if (chunkSize > stream.ByteCount()) {
		LogErrorMessage("font file is truncated");
		return false;
	}

	while (stream.ByteCount() > 0) {
		cookie    = stream.ReadFourCC();
		chunkSize = stream.ReadU32();

		if (FourCC_Header == cookie) {
			// header is currently ignored
		}
		else if (FourCC_Font == cookie) {
			QzByteStream subStream(stream.Address(), chunkSize);

			if (false == ScanFont(subStream, firstPass)) {
				return false;
			}
		}
		else {
			UtfFormat fmt;
			fmt.AddFourCC(cookie);
			LogErrorMessage("font file contains unknown chunk \"%1;\"", fmt);
		}

		stream.SkipData(chunkSize);
	}

	return stream.ChunkCleanup();
}


/////////////////////////////////////////////////////////////////////////////
//
//	ScanFont()
//
//	Scans through the character chunks stored within a font chunk.
//
bool QzFontHandler::ScanFont(QzByteStream &stream, bool firstPass)
{
	U32 fontID = stream.ReadU08();

	if (fontID >= QzFont_ArraySize) {
		UtfFormat fmt;
		fmt.AddInt(fontID);
		LogErrorMessage("font ID = %1; is out of range", fmt);
		return false;
	}

	stream.ReadString8(m_FontList[fontID].Label, ArraySize(m_FontList[fontID].Label));

	m_FontList[fontID].MaxWidth  = stream.ReadU08();
	m_FontList[fontID].MaxHeight = stream.ReadU08();

	U32 symbolCount = stream.ReadU32();

	// On the first pass, we need to allocate all of the buffers to hold the
	// font info about to be read in.
	if (firstPass) {
		if (NULL != m_FontList[fontID].pSymbols) {
			UtfFormat fmt;
			fmt.AddInt(fontID);
			LogErrorMessage("font ID = %1; was duplicated", fmt);
			return false;
		}

		if (symbolCount > 0) {
			m_FontList[fontID].SymbolCount = symbolCount;
			m_FontList[fontID].pSymbols    = new QzFontChar_t[symbolCount];
			memset(m_FontList[fontID].pSymbols, 0, symbolCount * sizeof(QzFontChar_t));
		}
	}

	// Extract the properties of each symbol and store it into the buffer.
	// Granted, on the second pass this will overwrite the values stored
	// there on the first pass, but both passes are processing the same
	// source data, so this should not make a difference.
	for (U32 symbolNum = 0; symbolNum < symbolCount; ++symbolNum) {
		U32 cookie    = stream.ReadFourCC();
		U32 chunkSize = stream.ReadU32();

		if (FourCC_Char != cookie) {
			UtfFormat fmt;
			fmt.AddFourCC(cookie);
			LogErrorMessage("font chunk contains unknown cookie %1;", fmt);
			return false;
		}

		if (chunkSize > stream.ByteCount()) {
			UtfFormat fmt;
			fmt.AddInt(symbolNum);
			LogErrorMessage("font symbol %1; chunk is truncated", fmt);
			return false;
		}

		QzByteStream subStream(stream.Address(), chunkSize);

		if (false == ScanSymbol(subStream, m_FontList[fontID].pSymbols[symbolNum], firstPass)) {
			return false;
		}

		stream.SkipData(chunkSize);
	}

	return stream.ChunkCleanup();
}


//////////////////////////////////////////////////////////////////////////////
//
//	DecodeLine08()
//
//	Reads a buffer full of data from the file, unpacking the RLE values back
//	into a full line of image data.  This is cheap and useful, since most of
//	the pixels in each symbol are zero.
//
static void DecodeLine08(U08 *pDst, U08 *pSrc, S32 pixelCount)
{
	// Loop until the pixel count hits zero.  If the pixel count should ever
	// attempt to go negative, sanity checking code will trap this and abort
	// reading the data, under the assumption that something is screwed up
	// with the file.
	while (pixelCount > 0) {
		U08 packet = *(pSrc++);

		// Determine how many pixels are in this block.
		S32 rleCount = S32(packet & 0x7F) + 1;
		pixelCount -= rleCount;

		// Trap any problems with the image data extending past the end of
		// buffer, which should never happen.  If it does, punt out and
		// assume the file is corrupt.
		if (pixelCount < 0) {
			return;
		}

		// If the high bit of this byte is set, then the block contains an
		// RLE-compressed pixel.  This means that the same color pixel
		// occurs rleCount times, so it needs to be read into a scratch
		// buffer, then copied that many times into the line buffer.
		if (packet & 0x80) {
			U08 bits = *(pSrc++);

			for ( ; rleCount > 0; --rleCount) {
				*(pDst++) = bits;
			}
		}

		// Handle the case of a block of raw pixel data.  Such a block
		// contains rleCount pixels, each different from the previous.
		// This is easily handled by reading the bytes directly into
		// the line buffer.
		else {
			for (S32 i = 0; i < rleCount; ++i) {
				pDst[i] = pSrc[i];
			}
			pDst += rleCount;
			pSrc += rleCount;
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ScanSymbol()
//
//	Extracts the contents of a symbol chunk into the given struct.
//
//	On the second pass, this assumes that the layout of the font texture has
//	been completed and that the PosX and PosY values have been computed for
//	each symbol's pixels, so it will decode the RLE data into the texture at
//	that position.
//
bool QzFontHandler::ScanSymbol(QzByteStream &stream, QzFontChar_t &info, bool firstPass)
{
	info.Symbol       = Utf16_t(stream.ReadU32());
	info.FullWidth    = stream.ReadU08();
	info.FullHeight   = stream.ReadU08();
	info.SpacingWidth = stream.ReadU08();
	info.DeltaX       = stream.ReadU08();
	info.ActiveWidth  = stream.ReadU08();
	info.ActiveHeight = stream.ReadU08();
	info.HiMidX       = stream.ReadU08();
	info.HiMidY       = stream.ReadU08();
	info.HiRightX     = stream.ReadU08();
	info.HiRightY     = stream.ReadU08();
	info.LoMidX       = stream.ReadU08();
	info.LoMidY       = stream.ReadU08();
	info.LoRightX     = stream.ReadU08();
	info.LoRightY     = stream.ReadU08();
	info.AttachX      = stream.ReadU08();
	info.AttachY      = stream.ReadU08();
	info.MinX         = stream.ReadU08();
	info.MinY         = stream.ReadU08();

	U32 encodedLength = stream.ReadU16();

	// On the second pass, decode the pixels into the texture.
	// Note that some symbols will not contain any pixels, although the only
	// such symbol should be ' ' -- that is the only intended whitespace
	// symbol.
	if ((false == firstPass) && (encodedLength > 0)) {
		// Allocate a scratch buffer, since the RLE data needs to be decoded
		// as a contiguous chunk of data.  Symbols are small, so we can safely
		// use alloca() to grab a chunk of memory from the stack.
		U08 *pScratch = reinterpret_cast<U08*>(alloca(info.ActiveWidth * info.ActiveHeight));
		DecodeLine08(pScratch, stream.Address(), info.ActiveWidth * info.ActiveHeight);

		U08 *pSrc = pScratch;
		U08 *pDst = m_pPixels + (info.PosY * m_ImageWidth) + info.PosX;

		// Now we need to blit the pixels into place one line at a time.
		for (U32 y = 0; y < U32(info.ActiveHeight); ++y) {
			for (U32 x = 0; x < U32(info.ActiveWidth); ++x) {
				pDst[x] = pSrc[x];
			}

			pSrc += info.ActiveWidth;
			pDst += m_ImageWidth;
		}
	}

	stream.SkipData(encodedLength);

	return stream.ChunkCleanup();
}


// Local struct only needed for sorting and packing the symbols into the
// texture buffer.
struct QzFontSortRef_t
{
	U32           Width;
	U32           Height;
	QzFontChar_t* pRef;
};


/////////////////////////////////////////////////////////////////////////////
//
//	PlaceSymbolsInTexture()
//
//	This will sort all of the symbols according to height, then pack them
//	into the texture buffer.
//
bool QzFontHandler::PlaceSymbolsInTexture(void)
{
	U32 symbolTotal = 0;
	for (U32 fontID = 0; fontID < QzFont_ArraySize; ++fontID) {
		symbolTotal += m_FontList[fontID].SymbolCount;
	}

	// This should not occur unless the font data file is corrupt.
	if (symbolTotal < 2) {
		LogErrorMessage("font file does not contain any symbols");
		return false;
	}

	QzFontSortRef_t*      pList  = new QzFontSortRef_t[symbolTotal];
	QzRectSortPosition_t* pRects = new QzRectSortPosition_t[symbolTotal];

	U32 symbolIndex = 0;

	for (U32 fontID = 0; fontID < QzFont_ArraySize; ++fontID) {
		QzFontInfo_t &info = m_FontList[fontID];

		for (U32 symbolNum = 0; symbolNum < info.SymbolCount; ++symbolNum) {
			pList[symbolIndex].Width  = info.pSymbols[symbolNum].ActiveWidth;
			pList[symbolIndex].Height = info.pSymbols[symbolNum].ActiveHeight;
			pList[symbolIndex].pRef   = &(info.pSymbols[symbolNum]);
			++symbolIndex;
		}
	}

	// Sort the list so that the tallest characters will be inserted first.
	// This can improve packing performance.
	for (U32 top = symbolIndex - 1; top > 0; --top) {
		for (U32 scan = 0; scan < top; ++scan) {
			if (pList[scan].Height < pList[top].Height) {
				QzFontSortRef_t temp = pList[scan];
				pList[scan]          = pList[top];
				pList[top]           = temp;
			}
		}
	}

	// The sorter assumes that the inputs are rectangles, and will perform a
	// simple, greedy subdivision approach.  Good performance is yielded when
	// rectangles are inserted from largest to smallest (as measured by the
	// longest dimension).
	QzRectSorter sorter;

	sorter.Initialize(m_ImageWidth, m_ImageHeight);

	bool success = true;

	for (U32 i = 0; i < symbolIndex; ++i) {
		pRects[i].Width    = pList[i].Width + 1;
		pRects[i].Height   = pList[i].Height + 1;
		pRects[i].pContext = &(pList[i]);
		pRects[i].pSort    = sorter.Insert(pRects[i]);

		// Can't insert.  Sorter cannot locate a subspace large enough for
		// this rectangle.  Abort this attempt and try again with a larger
		// target buffer.
		if (NULL == pRects[i].pSort) {
			LogErrorMessage("texture too small to contain all font characters");
			success = false;
			break;
		}
		else {
			pList[i].pRef->PosX = U16(pRects[i].pSort->m_Position.Left);
			pList[i].pRef->PosY = U16(pRects[i].pSort->m_Position.Top);
		}
	}

	SafeDeleteArray(pRects);
	SafeDeleteArray(pList);

	return success;
}


/////////////////////////////////////////////////////////////////////////////
//
//	LookUpChar()
//
//	This will start off with a binary look-up for the requested character.
//	The assumed normal case is that the character already exists, so the
//	appropriate index into m_pRects[] will be returned.
//
//	If the input string was properly compused with one of the UtfCompose...()
//	functions, the string will only contain known symbols, all of which have
//	glyphs in the font file.
//
//	If the character cannot be found (assume that this is an error in
//	UTF composition, or more likely that the string was not composed, or is
//	corrupt).
//
//	This will always return a valid pointer.  If the symbols no not in the
//	font, it will return the default '?' glyph.  At worst, it will return
//	the first symbol in the font, under the assumption that there is at
//	least one symbol.
//
QzFontChar_t* QzFontHandler::LookUpChar(U32 fontID, Utf32_t c)
{
	QzFontInfo_t &font = m_FontList[fontID];

	// FIXME: construct hash tables to speed this up

	//
	// Do a binary look-up over the contents of the list.
	//

	S32 lo  = 0;
	S32 hi  = S32(font.SymbolCount) - 1;

	while (lo <= hi) {
		S32 mid = (lo + hi) / 2;

		if (c < font.pSymbols[mid].Symbol) {
			hi = mid - 1;
		}
		else if (c > font.pSymbols[mid].Symbol) {
			lo = mid + 1;
		}
		else {
			return &(font.pSymbols[mid]);
		}
	}

	// If the symbol was not found, return the 0xFFFD symbol, which is used
	// to represent unknow glyphs -- but don't do that if the symbol in
	// question is 0xFFFD, otherwise we end up in infinite recursion if
	// 0xFFFD is not defined in this font.
	if (Unicode_UnknownSymbol != c) {
		return LookUpChar(fontID, Unicode_UnknownSymbol);
	}

	// Failsafe: return whatever happens to be the first symbol in the font.
	// We're assuming that if the font has not symbols, the font file is bad
	// and LoadFontFile() will have rejected the file, so the app should
	// never get to this point if there are no symbols in the current font.
	return &(font.pSymbols[0]);
}


/////////////////////////////////////////////////////////////////////////////
//
//	CharWidth()
//
//	Report the full width of the symbol, not the active width.  This is used
//	for spacing, so we don't care if some of the pixels are not drawn.
//
U32 QzFontHandler::CharWidth(U32 fontID, Utf32_t c)
{
	U32 maxWidth = 0;

	UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(c);

	// Each symbol is formed by 1, 2, or 3 different glyphs, but only the
	// base glyph is considered for computing the width of the whole glyph.
	if (NULL != pSymbol) {
		QzFontChar_t *pChar = LookUpChar(fontID, pSymbol->Render1);

		// Return the width of the character used for spacing purposes.
		// Normally, this is the width of the glyph, except in the case of
		// italic fonts, which may be overlapped to achieve better spacing.
		maxWidth = pChar->SpacingWidth;

		// Note that we're ignoring the two diacritics that may be attached
		// to the symbol.  The diacritics are not used for spacing, only the
		// base glyph.
	}

	return maxWidth;
}


/////////////////////////////////////////////////////////////////////////////
//
//	FontHeight()
//
U32 QzFontHandler::FontHeight(U32 fontID)
{
	return m_FontList[fontID].MaxHeight;
}


/////////////////////////////////////////////////////////////////////////////
//
//	LimitCharsByWidth()
//
//	Given a string and the maximum number of pixels allowed to display it,
//	this will determine how many of the chars will be visible within the
//	allowed space.  This will only include fully visible chars.  If the
//	last char is only partially visible, it will be ignored.
//
//	Note: Returns the number of UTF-32 chars processed.
//
U32 QzFontHandler::LimitCharsByWidth(U32 fontID, const Utf32_t strg[], U32 charCount, U32 maxWidth)
{
	U32 pixelCount = 0;
	U32 charNum    = 0;

	while (charNum < charCount) {
		if ('\0' == strg[charNum]) {
			break;
		}

		U32 width = CharWidth(fontID, strg[charNum]);

		// Return the previous char count, not the current one, otherwise
		// we'll try to draw the last, partially visible character.
		if ((pixelCount + width) > maxWidth) {
			return charNum;
		}

		pixelCount += width;
		charNum    += 1;
	}

	return charNum;
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetFontExtents()
//
//	Given a string, compute the size of the rectangle required to hold the
//	rendered string.
//
void QzFontHandler::GetFontExtents(U32 fontID, const Utf32_t strg[], U32 charCount, U32 &width, U32 &height)
{
	width  = 0;
	height = 0;

	QzFontInfo_t &font = m_FontList[fontID];

	height = font.MaxHeight;

	for (U32 i = 0; i < charCount; ++i) {
		if ('\0' == strg[i]) {
			break;
		}

		width += CharWidth(fontID, strg[i]);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetFontExtents()
//
//	Given a string, compute the size of the rectangle required to hold the
//	rendered string.
//
void QzFontHandler::GetFontExtents(U32 fontID, const Utf08_t strg[], U32 byteCount, U32 &width, U32 &height)
{
	width  = 0;
	height = 0;

	QzFontInfo_t &font = m_FontList[fontID];

	height = font.MaxHeight;

	U32 strgOffset = 0;

	// Limit the loop according to byte count.  There may be more chars in
	// the buffer that need to be ignored due to rendering reasons.
	while (strgOffset < byteCount) {
		Utf32_t symbol = UtfNextChar(strg, strgOffset);

		if ('\0' == symbol) {
			break;
		}

		width += CharWidth(fontID, symbol);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ComputeQuadCount()
//
//	Scan through the given string to determine how many quads will be needed
//	to render it.  Some chars won't need any quads (like spaces), while
//	others with accents will require two or three quads.
//
U32 QzFontHandler::ComputeQuadCount(U32 fontID, const Utf32_t strg[], U32 charCount)
{
	U32 quadCount = 0;

	for (U32 i = 0; i < charCount; ++i) {
		UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(strg[i]);

		// Each symbol is formed by 1, 2, or 3 different glyphs.  We need to
		// check each glyph to used to compuse the full symbol and verify
		// that it has a non-zero width (there should never be one that does
		// have a zero width, but this code needs to handle fonts that have
		// not yet been fully painted).
		if (NULL != pSymbol) {
			QzFontChar_t *pChar = LookUpChar(fontID, pSymbol->Render1);

			if (pChar->ActiveWidth > 0) {
				quadCount += 1;
			}

			if (0 != pSymbol->Render2) {
				pChar = LookUpChar(fontID, pSymbol->Render2);

				if (pChar->ActiveWidth > 0) {
					quadCount += 1;
				}

				if (0 != pSymbol->Render3) {
					pChar = LookUpChar(fontID, pSymbol->Render2);

					if (pChar->ActiveWidth > 0) {
						quadCount += 1;
					}
				}
			}
		}
	}

	return quadCount;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DrawGlyph()
//
void QzFontHandler::DrawGlyph(U32 x, U32 y, U32 width, U32 height, float u0, float u1, float v0, float v1)
{
	if (0 == m_GlyphCount) {
		glBegin(GL_TRIANGLES);
	}

	++m_GlyphCount;

	// FIXME: vertex buffers

	glTexCoord2f(u0, v0);
	glVertex2f(float(x), float(y));

	glTexCoord2f(u1, v0);
	glVertex2f(float(x + width), float(y));

	glTexCoord2f(u1, v1);
	glVertex2f(float(x + width), float(y + height));

	glTexCoord2f(u1, v1);
	glVertex2f(float(x + width), float(y + height));

	glTexCoord2f(u0, v1);
	glVertex2f(float(x), float(y + height));

	glTexCoord2f(u0, v0);
	glVertex2f(float(x), float(y));
}


/////////////////////////////////////////////////////////////////////////////
//
//	AssemblePixels()
//
//	Basic version that just draws characters on the screen.
//
U32 QzFontHandler::AssemblePixels(S32 x, S32 y, U32 color, U32 fontID, const Utf32_t strg[], U32 charCount)
{
	m_GlyphCount = 0;

	glColor3f(	float((color >> 16) & 0xFF) / 255.0f,
				float((color >>  8) & 0xFF) / 255.0f,
				float((color      ) & 0xFF) / 255.0f);

	const float scaleU = 1.0f / float(m_ImageWidth);
	const float scaleV = 1.0f / float(m_ImageHeight);

	U32 charNum = 0;

	// Limit the loop according to char count.  There may be more chars in
	// the buffer that need to be ignored due to rendering reasons.
	while (charNum < charCount) {
		Utf32_t symbol = strg[charNum++];

		// End of string.
		if ('\0' == symbol) {
			break;
		}

		// Assuming a properly composed string, this should never return NULL.
		UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(symbol);

		// If the symbol is not in the data tables, try using 0xFFFD.  But be
		// careful, since that symbol may also be missing.
		if (NULL == pSymbol) {
			symbol = Unicode_UnknownSymbol;
			UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(symbol);
			if (NULL == pSymbol) {
				continue;
			}
		}

		QzFontChar_t *pBaseChar = LookUpChar(fontID, pSymbol->Render1);

		if (NULL == pBaseChar) {
			continue;
		}

		S32 baseX = x - S32(pBaseChar->DeltaX);

		// Always check active height.  Some characters may not have any
		// visible pixels -- ' ' should be the only such character.
		if (pBaseChar->ActiveHeight > 0) {
			DrawGlyph(	baseX + U32(pBaseChar->MinX),
						y     + U32(pBaseChar->MinY),
						pBaseChar->ActiveWidth,
						pBaseChar->ActiveHeight,
						scaleU * float(pBaseChar->PosX),
						scaleU * float(pBaseChar->PosX + U32(pBaseChar->ActiveWidth)),
						scaleV * float(pBaseChar->PosY),
						scaleV * float(pBaseChar->PosY + U32(pBaseChar->ActiveHeight)));
		}

		// Now loop over the two diacritics that may be present.

		U32 diacID[2];
		diacID[0] = pSymbol->Render2;
		diacID[1] = pSymbol->Render3;

		S32 skipTop    = 0;
		S32 skipBottom = 0;


		for (U32 diacNum = 0; diacNum < 2; ++diacNum) {
			if (0 != diacID[diacNum]) {
				QzFontChar_t     *pDiac = LookUpChar(fontID, diacID[diacNum]);
				UtfSymbolTable_t *pSym  = UtfLookUpSymbol(diacID[diacNum]);

				if ((NULL != pDiac) && (NULL != pSym)) {
					S32 deltaX = S32(pDiac->MinX) - S32(pDiac->AttachX);
					S32 deltaY = S32(pDiac->MinY) - S32(pDiac->AttachY);

					if (0 != (UtfFlag_CombineHiMid & pSym->Flags)) {
						deltaX  += S32(pBaseChar->HiMidX);
						deltaY  += S32(pBaseChar->HiMidY) - skipTop;
						skipTop += S32(pDiac->AttachY) - S32(pDiac->MinY) + 1;
					}
					else if (0 != (UtfFlag_CombineHiRight & pSym->Flags)) {
						deltaX  += S32(pBaseChar->HiRightX);
						deltaY  += S32(pBaseChar->HiRightY) - skipTop;
						skipTop += S32(pDiac->AttachY) - S32(pDiac->MinY) + 1;
					}
					else if (0 != (UtfFlag_CombineLoMid & pSym->Flags)) {
						deltaX     += S32(pBaseChar->LoMidX);
						deltaY     += S32(pBaseChar->LoMidY) + skipBottom;
						skipBottom += S32(pDiac->MinY + pDiac->ActiveHeight) - S32(pDiac->AttachY) + 1;
					}
					else if (0 != (UtfFlag_CombineLoRight & pSym->Flags)) {
						deltaX     += S32(pBaseChar->LoRightX);
						deltaY     += S32(pBaseChar->LoRightY) + skipBottom;
						skipBottom += S32(pDiac->MinY + pDiac->ActiveHeight) - S32(pDiac->AttachY) + 1;
					}

					DrawGlyph(	baseX + deltaX,
								y     + deltaY,
								pBaseChar->ActiveWidth,
								pBaseChar->ActiveHeight,
								scaleU * float(pDiac->PosX),
								scaleU * float(pDiac->PosX + U32(pDiac->ActiveWidth)),
								scaleV * float(pDiac->PosY),
								scaleV * float(pDiac->PosY + U32(pDiac->ActiveHeight)));
				}
			}
		}

		x += U32(pBaseChar->SpacingWidth);
	}

	if (0 != m_GlyphCount) {
		glEnd();
	}

	return m_GlyphCount;
}


/////////////////////////////////////////////////////////////////////////////
//
//	AssemblePixelsCropped()
//
U32 QzFontHandler::AssemblePixelsCropped(S32 baseX, S32 baseY, const QzRect &crop, U32 color, U32 fontID, const Utf32_t strg[], U32 charCount)
{
	m_GlyphCount = 0;

	glColor3f(	float((color >> 16) & 0xFF) / 255.0f,
				float((color >>  8) & 0xFF) / 255.0f,
				float((color      ) & 0xFF) / 255.0f);

	const float scaleU = 1.0f / float(m_ImageWidth);
	const float scaleV = 1.0f / float(m_ImageHeight);

	U32 charNum = 0;

	while (charNum < charCount) {
		Utf32_t symbol = strg[charNum++];

		if ('\0' == symbol) {
			break;
		}

		UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(symbol);

		if (NULL == pSymbol) {
			symbol = Unicode_UnknownSymbol;
			UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(symbol);
			if (NULL == pSymbol) {
				continue;
			}
		}

		QzFontChar_t *pBaseChar = LookUpChar(fontID, pSymbol->Render1);

		if (NULL == pBaseChar) {
			continue;
		}

		// Use a copy, not a reference, so we can change its values locally.
		QzRect rect;

		rect.m_Left   = U32(pBaseChar->PosX);
		rect.m_Right  = U32(pBaseChar->PosX) + U32(pBaseChar->ActiveWidth);
		rect.m_Top    = U32(pBaseChar->PosY);
		rect.m_Bottom = U32(pBaseChar->PosY) + U32(pBaseChar->ActiveHeight);

		S32 x = baseX + S32(pBaseChar->MinX) - S32(pBaseChar->DeltaX);
		S32 y = baseY + S32(pBaseChar->MinY);

		if ((x + rect.Width()) > crop.m_Right) {
			rect.m_Right -= (x + rect.Width()) - crop.m_Right;
		}

		if ((y + rect.Height()) > crop.m_Bottom) {
			rect.m_Bottom -= (y + rect.Height()) - crop.m_Bottom;
		}

		if (x < crop.m_Left) {
			S32 delta = crop.m_Left - x;
			x += delta;
			rect.m_Left += delta;
		}

		if (y < crop.m_Top) {
			S32 delta = crop.m_Top - y;
			y += delta;
			rect.m_Top += delta;
		}

		if ((rect.Width() > 0) && (rect.Height() > 0)) {
			DrawGlyph(	x,
						y,
						rect.Width(),
						rect.Height(),
						scaleU * float(rect.m_Left),
						scaleU * float(rect.m_Right),
						scaleV * float(rect.m_Top),
						scaleV * float(rect.m_Bottom));
		}

		// FIXME: diacritics

		baseX += pBaseChar->SpacingWidth;
	}

	if (0 != m_GlyphCount) {
		glEnd();
	}

	return m_GlyphCount;
}


/////////////////////////////////////////////////////////////////////////////
//
//	AssemblePixelsFormatted()
//
U32 QzFontHandler::AssemblePixelsFormatted(S32 &posX, S32 &posY, const QzRect &pos, U32 pColors[], U08 pFontID[], const Utf32_t strg[], U32 charCount)
{
	m_GlyphCount = 0;

	const float scaleU = 1.0f / float(m_ImageWidth);
	const float scaleV = 1.0f / float(m_ImageHeight);

	U32 charNum = 0;

	while (charNum < charCount) {
		U32     color  = pColors[charNum];
		U32     fontID = pFontID[charNum];
		Utf32_t symbol = strg[charNum++];

		// End of string.
		if ('\0' == symbol) {
			break;
		}

		// Assuming a properly composed string, this should never return NULL.
		UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(symbol);

		// If the symbol is not in the data tables, try using 0xFFFD.  But be
		// careful, since that symbol may also be missing.
		if (NULL == pSymbol) {
			symbol = Unicode_UnknownSymbol;
			UtfSymbolTable_t *pSymbol = UtfLookUpSymbol(symbol);
			if (NULL == pSymbol) {
				continue;
			}
		}

		QzFontChar_t *pBaseChar = LookUpChar(fontID, pSymbol->Render1);

		if (NULL == pBaseChar) {
			continue;
		}

		S32 x = posX + pos.m_Left;
		S32 y = posY + pos.m_Top;

		S32 baseX = x - S32(pBaseChar->DeltaX);

		// Don't use byte version of glColor -- it's buggy on my graphics card.
		glColor3f(	float((color >> 16) & 0xFF) / 255.0f,
					float((color >>  8) & 0xFF) / 255.0f,
					float((color      ) & 0xFF) / 255.0f);


		// Always check active height.  Some characters may not have any
		// visible pixels -- ' ' should be the only such character.
		if (pBaseChar->ActiveHeight > 0) {
			DrawGlyph(	baseX + U32(pBaseChar->MinX),
						y     + U32(pBaseChar->MinY),
						pBaseChar->ActiveWidth,
						pBaseChar->ActiveHeight,
						scaleU * float(pBaseChar->PosX),
						scaleU * float(pBaseChar->PosX + U32(pBaseChar->ActiveWidth)),
						scaleV * float(pBaseChar->PosY),
						scaleV * float(pBaseChar->PosY + U32(pBaseChar->ActiveHeight)));
		}

		U32 diacID[2];
		diacID[0] = pSymbol->Render2;
		diacID[1] = pSymbol->Render3;

		S32 skipTop    = 0;
		S32 skipBottom = 0;

		for (U32 diacNum = 0; diacNum < 2; ++diacNum) {
			if (0 != diacID[diacNum]) {
				QzFontChar_t     *pDiac = LookUpChar(fontID, diacID[diacNum]);
				UtfSymbolTable_t *pSym  = UtfLookUpSymbol(diacID[diacNum]);

				if ((NULL != pDiac) && (NULL != pSym)) {
					S32 deltaX = S32(pDiac->MinX) - S32(pDiac->AttachX);
					S32 deltaY = S32(pDiac->MinY) - S32(pDiac->AttachY);

					if (0 != (UtfFlag_CombineHiMid & pSym->Flags)) {
						deltaX  += S32(pBaseChar->HiMidX);
						deltaY  += S32(pBaseChar->HiMidY) - skipTop;
						skipTop += S32(pDiac->AttachY) - S32(pDiac->MinY) + 1;
					}
					else if (0 != (UtfFlag_CombineHiRight & pSym->Flags)) {
						deltaX  += S32(pBaseChar->HiRightX);
						deltaY  += S32(pBaseChar->HiRightY) - skipTop;
						skipTop += S32(pDiac->AttachY) - S32(pDiac->MinY) + 1;
					}
					else if (0 != (UtfFlag_CombineLoMid & pSym->Flags)) {
						deltaX     += S32(pBaseChar->LoMidX);
						deltaY     += S32(pBaseChar->LoMidY) + skipBottom;
						skipBottom += S32(pDiac->MinY + pDiac->ActiveHeight) - S32(pDiac->AttachY) + 1;
					}
					else if (0 != (UtfFlag_CombineLoRight & pSym->Flags)) {
						deltaX     += S32(pBaseChar->LoRightX);
						deltaY     += S32(pBaseChar->LoRightY) + skipBottom;
						skipBottom += S32(pDiac->MinY + pDiac->ActiveHeight) - S32(pDiac->AttachY) + 1;
					}

					DrawGlyph(	baseX + deltaX,
								y     + deltaY,
								pDiac->ActiveWidth,
								pDiac->ActiveHeight,
								scaleU * float(pDiac->PosX),
								scaleU * float(pDiac->PosX + U32(pDiac->ActiveWidth)),
								scaleV * float(pDiac->PosY),
								scaleV * float(pDiac->PosY + U32(pDiac->ActiveHeight)));
				}
			}
		}

		posX += U32(pBaseChar->SpacingWidth);
	}

	if (0 != m_GlyphCount) {
		glEnd();
	}

	return m_GlyphCount;
}




