//////////////////////////////////////////////////////////////////////////////
//
//	File: QzFile.cpp
//
//	$Header: /TS/TsFile/QzFile.cpp  16  2009/9/9 4:16:07p  Lee $
//
//
//	Simple wrapper class around FILE routines to simplify read/write calls,
//	and project against files not being closed.
//
//////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzFile.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


//////////////////////////////////////////////////////////////////////////////
//
QzFile::QzFile(void)
	:	m_Writeable(false),
		m_pFile(NULL)
{
}


//////////////////////////////////////////////////////////////////////////////
//
QzFile::~QzFile(void)
{
	Close();
}


//////////////////////////////////////////////////////////////////////////////
//
//	CreateNew()
//
//	Returns true if the file was successfully opened.
//
bool QzFile::CreateNew(const Utf08_t filename[])
{
	Close();

	if ((NULL == filename) || ('\0' == filename[0])) {
		return false;
	}

	m_Writeable = true;

	m_pFile = QzFileOpen(filename, QzFileOpen_Write);

	return (NULL != m_pFile);
}


//////////////////////////////////////////////////////////////////////////////
//
//	OpenExisting()
//
//	Returns true if the file was successfully opened.
//
bool QzFile::OpenExisting(const Utf08_t filename[])
{
	Close();

	if ((NULL == filename) || ('\0' == filename[0])) {
		return false;
	}

	m_Writeable = false;

	m_pFile = QzFileOpen(filename, QzFileOpen_Read);

	return (NULL != m_pFile);
}


//////////////////////////////////////////////////////////////////////////////
//
//	Close()
//
//	Closes the current file if one is open.
//
void QzFile::Close(void)
{
	if (NULL != m_pFile) {
		fclose(m_pFile);
		m_pFile = NULL;
	}
}


//////////////////////////////////////////////////////////////////////////////
//
//	ReadData()
//
//	Reads a chunk of data from an input file, returning true if all of the
//	data was successfully read in.
//
bool QzFile::ReadData(void *pData, U32 byteCount)
{
	if (IsOpen() && (false == m_Writeable)) {
		return (byteCount == fread(pData, 1, byteCount, m_pFile));
	}

	return false;
}


//////////////////////////////////////////////////////////////////////////////
//
//	WriteData()
//
//	Writes a chunk of data to an output file, returning true if all of the
//	data was successfully written.
//
bool QzFile::WriteData(void *pData, U32 byteCount)
{
	if (IsOpen() && m_Writeable) {
		return (byteCount == fwrite(pData, 1, byteCount, m_pFile));
	}

	return false;
}


//////////////////////////////////////////////////////////////////////////////
//
//	Seek()
//
//	Seeks to a position relative of the start of the file.
//
bool QzFile::Seek(U32 offset)
{
	if (IsOpen()) {
		fseek(m_pFile, offset, SEEK_SET);

		return true;
	}

	return false;
}


//////////////////////////////////////////////////////////////////////////////
//
//	SeekToEnd()
//
//	Sets the file pointer to the end of the file.
//
bool QzFile::SeekToEnd(void)
{
	if (IsOpen()) {
		fseek(m_pFile, 0, SEEK_END);

		return true;
	}

	return false;
}


//////////////////////////////////////////////////////////////////////////////
//
//	Position()
//
//	Returns the current file position relative to the start of the file.
//
U32 QzFile::Position(void)
{
	if (false == IsOpen()) {
		return 0;
	}

	U32 offset = ftell(m_pFile);

	return offset;
}


//////////////////////////////////////////////////////////////////////////////
//
//	Size()
//
//	Computes the total size of the file.  The most reliable, portable way of
//	doing this is to seek to the end of the file, check the position, then
//	seek back.
//
U32 QzFile::Size(void)
{
	if (false == IsOpen()) {
		return 0;
	}

	U32 offset = ftell(m_pFile);

	fseek(m_pFile, 0, SEEK_END);

	U32 size = ftell(m_pFile);

	fseek(m_pFile, offset, SEEK_SET);

	return size;
}

