/////////////////////////////////////////////////////////////////////////////
//
//	File: QzDirList.cpp
//
//	$Header: /Projects/Qz/QzDirListMac.cpp  4  2009/9/7 1:11:48p  Lee $
//
//
//	This is a wrapper class around the native file iteration logic.  It will
//	scan a given directory, extracting all folders and files (optionally
//	filtering only files that have a specific extension).
//
//	The resulting list will be sorted alphabetically (case in-sensitive).
//	By default, the sorting will list all folder names first, followed by
//	the file names.  Clear the m_SortDirsFirst flag to force file and folder
//	names to be intermixed.
//
//	Make certain the desired m_Show... flags are set before making the first
//	call to ScanDirectory().  The m_Show... flags all default to false, so
//	forgetting to set them will result in an empty list.
//
//	After the initial call to ScanDirectory(), simple navigation to other
//	folders can be accomplished by calling UpOneLevel() and DownOneLevel().
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "UtfString.h"
#include "QzDirList.h"
#include <dirent.h>
#include <sys/stat.h>


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzDirList::QzDirList(void)
	:	m_ShowDirs(false),
		m_ShowFiles(false),
		m_ShowHidden(false),
		m_ShowReadOnly(false),
		m_ShowSystem(false),
		m_SortDirsFirst(true),
		m_EntryCount(0),
		m_MaxEntryCount(1024),
		m_pList(NULL)
{
	// Pre-allocate a large buffer.  If it ends up being too small, we can
	// grow it at run-time.
	m_pList        = new QzDirEntry_t[m_MaxEntryCount];
	m_Path[0]      = '\0';
	m_Extension[0] = '\0';
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzDirList::~QzDirList(void)
{
	Free();

	SafeDeleteArray(m_pList);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Free()
//
//	Frees all of the file names in the array.  The file names are dynamically
//	allocated since they can be of arbitrary length.
//
void QzDirList::Free(void)
{
	for (U32 i = 0; i < m_EntryCount; ++i) {
		SafeDeleteArray(m_pList[i].pName);
	}

	m_EntryCount = 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Grow()
//
//	Reallocate the buffer containing the array of files/folders.
//
void QzDirList::Grow(void)
{
	if (m_EntryCount >= m_MaxEntryCount) {
		QzDirEntry_t *pNew = new QzDirEntry_t[2 * m_MaxEntryCount];

		for (U32 i = 0; i < m_EntryCount; ++i) {
			pNew[i] = m_pList[i];

			m_pList[i].pName = NULL;
		}

		SafeDeleteArray(m_pList);

		m_pList          = pNew;
		m_MaxEntryCount *= 2;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	SortRange()
//
//	Sorts a range of entries in the array.  This allows the array to be
//	partitioned into separate groups, typically with the folder names and
//	files names in two separate groups.
//
//	Note: <hi> is the index of the last element to sort.  The total number
//	of elements is ((hi - lo) + 1).
//
void QzDirList::SortRange(S32 lo, S32 hi)
{
	// Stop if there is only one (or zero) element to sort.
	if ((hi - lo) < 1) {
		return;
	}

	// Do a bubble sort for simplicity.  Merge sort would be a much better
	// choice, since the file names returned by the OS may already be
	// sorted.  Note that quicksort is a bad algorithm here since it performs
	// so very poorly when given sorted data.
	for (S32 tail = hi; tail > lo; --tail) {
		for (S32 scan = lo; scan < tail; ++scan) {
			if (UtfCompareLexical(m_pList[scan].pName, m_pList[tail].pName) > 0) {
				Swap(m_pList[scan], m_pList[tail]);
			}
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ScanPath()
//
//	This is the private method that does the actual file enumeration. It
//	should only be called after m_Path has been populated with a correctly
//	formatted file path.
//
bool QzDirList::ScanPath(void)
{
	Free();

	Utf08_t tempName[c_MaxPathLength];
	Utf08_t tempPath[c_MaxPathLength];

	DIR *pDir = opendir(reinterpret_cast<char*>(m_Path));

	// NOTE: m_ShowReadOnly is ignored in this code.  Supporting it requires
	// reading the bits in statInfo.st_attr, which has different bits for
	// owner, group, and others.  Which of those bits should be tested for
	// read-only permission?  That's more of a policy question than in
	// implementation issue, so it has been left out of this code.

	if (NULL != pDir) {
		dirent*     pInfo;
		struct stat statInfo;

		// Note: In a multithreading environment, readdir_r is recommended
		// instead of readdir.  It is also considered harmful by some
		// programmers, since it presents posibilities for race conditions
		// and buffer overruns.  Do some research first before using readdir_r.

		while (NULL != (pInfo = readdir(pDir))) {

			// Ignore hidden files.
			// For Mac/Linux, file names starting with a period will normally
			// be hidden in any directory listing.
			if ((false == m_ShowHidden) && ('.' == pInfo->d_name[0])) {
			}
			else if (DT_DIR != pInfo->d_type) {
				if (m_ShowFiles) {
					bool addMe = true;

					// Note: Convert file names, do not compose.  Composing
					// would potentially screw up the name of the file.
					SafeStrCopy(tempName, reinterpret_cast<Utf08_t*>(pInfo->d_name));

					// By default, all files will be added to the list.
					// However, if a file extension
					if ('\0' != m_Extension[0]) {
						if (false == UtfSuffixNocase(m_Extension, tempName)) {
							addMe = false;
						}
					}

					if (addMe) {
						Utf08_t *pNew = QzAllocString(tempName);

						Grow();

						m_pList[m_EntryCount].IsDir = false;
						m_pList[m_EntryCount].pName = pNew;
						m_pList[m_EntryCount].Size  = 0;
						m_pList[m_EntryCount].Time  = 0;

						UtfCopy(tempPath, ArraySize(tempPath), m_Path);
						UtfAppend(tempPath, ArraySize(tempPath), CharToUtf(pInfo->d_name));

						// Need to directly access the file to find out the size and
						// timestamp info.
						if (0 == stat(reinterpret_cast<char*>(tempPath), &statInfo)) {
							m_pList[m_EntryCount].Size = statInfo.st_size;
							m_pList[m_EntryCount].Time = statInfo.st_mtime;
						}

						++m_EntryCount;
					}
				}
			}
			else if (UtfCompareBytewise(reinterpret_cast<Utf08_t*>(pInfo->d_name), reinterpret_cast<const Utf08_t*>(".")) &&
					 UtfCompareBytewise(reinterpret_cast<Utf08_t*>(pInfo->d_name), reinterpret_cast<const Utf08_t*>("..")))
			{
				if (m_ShowDirs) {
					// Note: Convert file names, do not compose.  Composing
					// would potentially screw up the name of the file.
					U32 byteCount = SafeStrCopy(tempName, reinterpret_cast<Utf08_t*>(pInfo->d_name));

					Utf08_t *pNew = new Utf08_t[byteCount + 2];
					memcpy(pNew, tempName, byteCount);
					pNew[byteCount  ] = '/';
					pNew[byteCount+1] = '\0';

					Grow();


					m_pList[m_EntryCount].IsDir = true;
					m_pList[m_EntryCount].pName = pNew;

					m_pList[m_EntryCount].Size  = 0;
					m_pList[m_EntryCount].Time  = 0;

					UtfCopy(tempPath, ArraySize(tempPath), m_Path);
					UtfAppend(tempPath, ArraySize(tempPath), CharToUtf(pInfo->d_name));

					// Need to directly access the file to find out the size and
					// timestamp info.
					if (0 == stat(reinterpret_cast<char*>(tempPath), &statInfo)) {
						m_pList[m_EntryCount].Size = statInfo.st_size;
						m_pList[m_EntryCount].Time = statInfo.st_mtime;
					}

					++m_EntryCount;
				}
			}
		}

		closedir(pDir);
	}

	// Optionally, the list can be sorted so all directory names come first,
	// followed by all file names.
	if (m_SortDirsFirst) {

		// Scan through the list, swapping as needed so that all of the
		// folders are at the start of the list.  When done, <nextDir> will
		// contain the total number of folders present.
		U32 nextDir = 0;
		for (U32 i = 0; i < m_EntryCount; ++i) {
			if (m_pList[i].IsDir) {
				Swap(m_pList[i], m_pList[nextDir]);
				++nextDir;
			}
		}

		// Alphabetically sort all of the folder names.
		SortRange(0, S32(nextDir) - 1);

		// Then sort all of the file names.
		SortRange(nextDir, S32(m_EntryCount) - 1);
	}
	else {
		SortRange(0, S32(m_EntryCount) - 1);
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ScanDirectory()
//
//	pExtension must either be NULL, or a dotted file extension like ".txt".
//	Do not use wildcards in the file extension -- "*.txt" would only confuse
//	the code.
//
//	All files will be added if pExtension is NULL.  Otherwise, only those
//	that have the given file extension will be added to the list.  The test
//	is case in-sensitive.
//
bool QzDirList::ScanDirectory(const Utf08_t path[], const Utf08_t *pExtension)
{
	if (NULL == pExtension) {
		m_Extension[0] = '\0';
	}
	else {
		SafeStrCopy(m_Extension, pExtension);
	}

	if (NULL == path) {
		m_Path[0] = '.';
		m_Path[1] = '/';
		m_Path[2] = '\0';
	}
	else {
		U32 byteCount = QzAbsolutePathOfFile(m_Path, ArraySize(m_Path), path);

		if ((byteCount + 2) >= ArraySize(m_Path)) {
			return false;
		}

		// Make certain that the path ends with a slash.  But only if the
		// string is not empty.  An empty string indicates that the code
		// should parse the list of drive names.
		if ((byteCount > 0) && ('/' != m_Path[byteCount-1])) {
			m_Path[byteCount  ] = '/';
			m_Path[byteCount+1] = '\0';
		}
	}

	return ScanPath();
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpOneLevel()
//
//	Move up to the parent of the current folder, then scan that folder.
//	This should only be called after a successful call to ScanDirectory().
//	It will preserve the filter extension.
//
//	Note that this is accomplished by trimming the last folder name off of
//	the absolute path.  Doing it this way avoids problems when traversing
//	down a soft link to another directory (simply moving to the parent of
//	the current directory will not always take you back to wherever you
//	previously where in the directory tree).
//
bool QzDirList::UpOneLevel(void)
{
	// Can't move up when sitting at the root level.
	if (('\0' == m_Path[0]) || (('/' == m_Path[0]) && ('\0' == m_Path[1]))) {
		return false;
	}

	// We need to scan backwards from the end of the string.  Find out the
	// offset of the last char.  Note we're using bytes, not chars.  Since
	// we're searching for a 7-bit ASCII string, we can safely ignore any
	// multi-byte UTF-8 chars in the string.
	S32  byteCount = UtfByteCount(m_Path);
	bool found     = false;

	// Starting at the end of the current path, clip off the last folder
	// name so we have the path of the parent directory.  Or an empty
	// string if we're at the root.
	//
	// Note that the path name normally ends in '/', so we need to ignore
	// that byte and start scanning from the byte right before it.  Again,
	// '/' is a 7-bit ASCII char, so we can safely ignore multi-byte
	// character (which always have the high bit set).
	//
	for (S32 i = byteCount - 2; i >= 0; --i) {
		if (('\\' == m_Path[i]) || ('/' == m_Path[i])) {
			found = true;
			m_Path[i+1] = '\0';
			break;
		}
	}

	// If the search failed, we're at the root of the directory structure.
	// Set the path to an empty string so that ScanPath() will know to
	// enumerate the contents of the root directory.
	if (false == found) {
		m_Path[0] = '\0';
	}

	return ScanPath();
}


/////////////////////////////////////////////////////////////////////////////
//
//	DownOneLevel()
//
//	Changes the current path to one of the child folders.  The <dirName>
//	should be the relative name of the child folder, not the full path.
//	This will concatenate <dirName> to the current path to create the new
//	absolute path.
//
//	Note that this code does not attempt to detect badly formatted
//	drive names.  It assumes that the name being given was created by
//	enumerating the contents of the current directory, and that dirName
//	points to a string stored somewhere in m_pList (since that is how
//	this class is used from a higher level).
//
bool QzDirList::DownOneLevel(const Utf08_t dirName[])
{
	// Use byte counts, not char counts.  We're going to concatenate the
	// path and directory name into a single buffer, so we need to know
	// the exact number of bytes to copy, and where to store them.
	U32 pathLen = UtfByteCount(m_Path);
	U32 newLen  = UtfByteCount(dirName);

	// Error out if the name is NULL or empty.
	if (0 == newLen) {
		return false;
	}

	// Is the name too long for the buffer?
	if ((pathLen + newLen + 2) >= ArraySize(m_Path)) {
		return false;
	}

	// Append the new directory name to the end of the path.
	memcpy(m_Path + pathLen, dirName, newLen);

	// Back slashes should never occur, but always protect against them.
	if ('\\' == dirName[newLen-1]) {
		// Replace the existing backslash.  We're not changing the length
		// of the string, so no extra terminator is required in this case.
		m_Path[pathLen+newLen-1] = '/';
	}

	// If the directory name did not already have a "/" at the end, append
	// one. This marker is expected by ScanPath(), which assumes any string
	// given to it was properly conditioned.
	else if ('/' != dirName[newLen-1]) {
		m_Path[pathLen+newLen  ] = '/';
		m_Path[pathLen+newLen+1] = '\0';
	}

	return ScanPath();
}



