/////////////////////////////////////////////////////////////////////////////
//
//	File: QzByteStream.h
//
//	$Header: /TS/TsFile/QzByteStream.cpp  19  2009/9/14 2:15:33p  Lee $
//
//
//	Utility class for extracting data from a packed byte stream.  This will
//	enforce protection against buffer overruns, silently returning zeroes if
//	there is not enough data to fulfill any given request.  If there is an
//	overflow, this will set an internal flag to signal the problem.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzByteStream.h"
#include "QzCRC32.h"
#include "QzQuat.h"
#include "QzVector.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzByteStream::QzByteStream(void)
	:	m_pData(NULL),
		m_ByteCount(0),
		m_Overflow(false)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzByteStream::QzByteStream(U08 *pData, U32 byteCount)
	:	m_pData(pData),
		m_ByteCount(byteCount),
		m_Overflow(false)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	StartStream()
//
//	Normally the buffer should be set by the constructor.  If not, use this
//	accessor to assign a new buffer to an existing stream object.
//
void QzByteStream::StartStream(U08 *pData, U32 byteCount)
{
	m_pData     = pData;
	m_ByteCount = byteCount;
	m_Overflow  = false;
}


/////////////////////////////////////////////////////////////////////////////
//
//	PeekU08()
//
U08 QzByteStream::PeekU08(void)
{
	if (m_ByteCount >= sizeof(U08)) {
		return m_pData[0];
	}

	m_Overflow = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	PeekU16()
//
U16 QzByteStream::PeekU16(void)
{
	if (m_ByteCount >= sizeof(U16)) {
		return U16(m_pData[0]) | (U16(m_pData[1]) << 8);
	}

	m_Overflow = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	PeekU32()
//
U32 QzByteStream::PeekU32(void)
{
	if (m_ByteCount >= sizeof(U32)) {
		return U32(m_pData[0]) | (U32(m_pData[1]) << 8) | (U32(m_pData[2]) << 16) | (U32(m_pData[3]) << 24);
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	SkipData()
//
void QzByteStream::SkipData(U32 byteCount)
{
	if (m_ByteCount >= byteCount) {
		m_pData     += byteCount;
		m_ByteCount -= byteCount;
	}
	else {
		m_ByteCount = 0;
		m_Overflow  = true;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadArrayU08()
//
void QzByteStream::ReadArrayU08(U08 *pData, U32 valueCount)
{
	if (valueCount <= m_ByteCount) {
		memcpy(pData, m_pData, valueCount);
		m_pData     += valueCount;
		m_ByteCount -= valueCount;
		return;
	}

	memset(pData, 0, valueCount);
	m_ByteCount = 0;
	m_Overflow  = true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadArrayU16()
//
void QzByteStream::ReadArrayU16(U16 *pData, U32 valueCount)
{
	if ((valueCount * sizeof(U16)) <= m_ByteCount) {
#ifdef IS_BIG_ENDIAN
		for (U32 i = 0; i < valueCount; ++i) {
			pData[i] = U16(m_pData[(i<<1)+1]) | (U16(m_pData[i<<1]) << 8);
		}
#else
		memcpy(pData, m_pData, valueCount * sizeof(U16));
#endif
		m_pData     += valueCount * sizeof(U16);
		m_ByteCount -= valueCount * sizeof(U16);
		return;
	}

	memset(pData, 0, valueCount * sizeof(U16));
	m_ByteCount = 0;
	m_Overflow  = true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadArrayU32()
//
void QzByteStream::ReadArrayU32(U32 *pData, U32 valueCount)
{
	if ((valueCount * sizeof(U32)) <= m_ByteCount) {
#ifdef IS_BIG_ENDIAN
		for (U32 i = 0; i < valueCount; ++i) {
			pData[i] = U32(m_pData[i<<2]) | (U32(m_pData[(i<<2)+1]) << 8) | (U32(m_pData[(i<<2)+2]) << 16) | (U32(m_pData[(i<<2)+3]) << 24);
		}
#else
		memcpy(pData, m_pData, valueCount * sizeof(U32));
#endif
		m_pData     += valueCount * sizeof(U32);
		m_ByteCount -= valueCount * sizeof(U32);
		return;
	}

	memset(pData, 0, valueCount * sizeof(U32));
	m_ByteCount = 0;
	m_Overflow  = true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadVarS32()
//
//	A signed var is the same as ReadVarU32(), except that it is stored in a
//	sign-magnitude format, with the LSB as the sign bit.
//
S32 QzByteStream::ReadVarS32(void)
{
	U32 t = ReadVarU32();

	// If the low bit is set, the value is negative.
	if (1 & t) {
		return -S32(t >> 1);
	}

	return S32(t >> 1);
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadVarU32()
//
//	This is packed as a sequence of 7-bit values.  If the high bit is set,
//	there are 7 more bits stored in the next byte.  Keep reading until we
//	reach a byte whose MSB is zero.
//
U32 QzByteStream::ReadVarU32(void)
{
	U32 shift  = 7;
	U32 temp   = ReadU08();
	U32 result = temp & 0x7F;

	// Continue reading bytes as long as the high bit is set.  The last byte
	// in the sequence will have the high bit cleared.
	while (0 != (0x80 & temp)) {
		temp    = ReadU08();
		result |= (temp & 0x7F) << shift;
		shift  += 7;
	}

	return result;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadU08()
//
U08 QzByteStream::ReadU08(void)
{
	if (m_ByteCount > 0) {
		--m_ByteCount;
		return *(m_pData++);
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadU16()
//
U16 QzByteStream::ReadU16(void)
{
	if (m_ByteCount >= sizeof(U16)) {
		U16 value    = U16(m_pData[0]) | (U16(m_pData[1]) << 8);
		m_pData     += sizeof(U16);
		m_ByteCount -= sizeof(U16);
		return value;
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadBeU16()
//
//	Reads 16-bit value that is stored in Big-Endian ordering.
//
U16 QzByteStream::ReadBeU16(void)
{
	if (m_ByteCount >= sizeof(U16)) {
		U16 value    = U16(m_pData[1]) | (U16(m_pData[0]) << 8);
		m_pData     += sizeof(U16);
		m_ByteCount -= sizeof(U16);
		return value;
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadU32()
//
U32 QzByteStream::ReadU32(void)
{
	if (m_ByteCount >= sizeof(U32)) {
		U32 value    = (U32(m_pData[0])      )
					 | (U32(m_pData[1]) <<  8)
					 | (U32(m_pData[2]) << 16)
					 | (U32(m_pData[3]) << 24);
		m_pData     += sizeof(U32);
		m_ByteCount -= sizeof(U32);
		return value;
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadBeU32()
//
//	Reads 32-bit value that is stored in Big-Endian ordering.
//
U32 QzByteStream::ReadBeU32(void)
{
	if (m_ByteCount >= sizeof(U32)) {
		U32 value    = (U32(m_pData[3])      )
					 | (U32(m_pData[2]) <<  8)
					 | (U32(m_pData[1]) << 16)
					 | (U32(m_pData[0]) << 24);
		m_pData     += sizeof(U32);
		m_ByteCount -= sizeof(U32);
		return value;
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadFourCC()
//
//	This is identical to ReadU32() on an Intel little-endian platform.  For
//	big-endian, this would need to leave the byte order alone, while
//	ReadU32() would need to reverse the byte-order (since QzMakeFourCC will
//	arrange bytes in the opposite order on big-endian systems).
//
U32 QzByteStream::ReadFourCC(void)
{
	if (m_ByteCount >= sizeof(U32)) {
		U32 value    = (U32(m_pData[0])      )
					 | (U32(m_pData[1]) <<  8)
					 | (U32(m_pData[2]) << 16)
					 | (U32(m_pData[3]) << 24);
		m_pData     += sizeof(U32);
		m_ByteCount -= sizeof(U32);
		return value;
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadFloat()
//
float QzByteStream::ReadFloat(void)
{
	if (m_ByteCount >= sizeof(float)) {
#ifdef NORMAL_ENDIAN_FLOATS
		U32 value    = (U32(m_pData[0])      )
					 | (U32(m_pData[1]) <<  8)
					 | (U32(m_pData[2]) << 16)
					 | (U32(m_pData[3]) << 24);
#else
		U32 value    = (U32(m_pData[3])      )
					 | (U32(m_pData[2]) <<  8)
					 | (U32(m_pData[1]) << 16)
					 | (U32(m_pData[0]) << 24);
#endif
		m_pData     += sizeof(float);
		m_ByteCount -= sizeof(float);
		return reinterpret_cast<float*>(&value)[0];
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadS16()
//
S16 QzByteStream::ReadS16(void)
{
	if (m_ByteCount >= sizeof(S16)) {
		S16 value    = S16(U16(m_pData[0]) | (U16(m_pData[1]) << 8));
		m_pData     += sizeof(S16);
		m_ByteCount -= sizeof(S16);
		return value;
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadS32()
//
S32 QzByteStream::ReadS32(void)
{
	if (m_ByteCount >= sizeof(S32)) {
		U32 value    = (U32(m_pData[0])      )
					 | (U32(m_pData[1]) <<  8)
					 | (U32(m_pData[2]) << 16)
					 | (U32(m_pData[3]) << 24);
		m_pData     += sizeof(S32);
		m_ByteCount -= sizeof(S32);
		return S32(value);
	}

	m_ByteCount = 0;
	m_Overflow  = true;

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadQuaternion()
//
void QzByteStream::ReadQuaternion(QzQuat &quat)
{
	quat.m_X = ReadFloat();
	quat.m_Y = ReadFloat();
	quat.m_Z = ReadFloat();
	quat.m_W = ReadFloat();
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadVector()
//
void QzByteStream::ReadVector(QzVector &vec)
{
	vec.m_X = ReadFloat();
	vec.m_Y = ReadFloat();
	vec.m_Z = ReadFloat();
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadString()
//
//	Read a UTF-8 string.  This assumes that the size of the string was
//	stored right before the data, and that the caller has already read in
//	that value and knows how many bytes to read.
//
//	bufferSize - exact size of buffer to hold the string
//	readSize   - number of bytes to extract from the stream
//
//	In the case where the buffer is too small, the string is truncated and
//	guaranteed to always be terminated.  The overflow data is skipped to
//	allow continued processing of the byte stream.
//
void QzByteStream::ReadString(Utf08_t *pBuffer, U32 bufferSize, U32 readSize)
{
	if (readSize < bufferSize) {
		ReadArrayU08(pBuffer, readSize);
		pBuffer[readSize] = '\0';
	}
	else {
		ReadArrayU08(pBuffer, bufferSize);
		pBuffer[bufferSize-1] = '\0';

		SkipData(readSize - bufferSize);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadString8()
//
//	This assumes that the first byte contains the number of bytes in
//	the string.  The length may be zero.  This is symmetrical to
//	QzByteAccumulator::WriteString8().
//
void QzByteStream::ReadString8(Utf08_t *pBuffer, U32 maxLength)
{
	U32 byteCount = ReadU08();

	// Avoid overflowing the destination buffer.
	U32 copyLength = Min(maxLength - 1, byteCount);

	ReadArrayU08(pBuffer, copyLength);

	// Zero out any remaining space in the buffer (since the string pulled
	// from the byte stream is not necessarily zero-terminated).
	for (U32 i = copyLength; i < maxLength; ++i) {
		pBuffer[i] = '\0';
	}

	// Skip any chars that could not be read in.
	if (byteCount > copyLength) {
		UtfFormat fmt;
		fmt.AddInt(byteCount);
		fmt.AddInt(maxLength);
		LogErrorMessage("ReadString8() field is %1; bytes, buffer is only %2; chars", fmt);
		SkipData(byteCount - copyLength);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadString16()
//
//	This assumes that the first 16-bit word contains the number of bytes in
//	the string.  The length may be zero.  This is symmetrical to
//	QzByteAccumulator::WriteString16().
//
void QzByteStream::ReadString16(Utf08_t *pBuffer, U32 maxLength)
{
	U32 byteCount = ReadU16();

	// Avoid overflowing the destination buffer.
	U32 copyLength = Min(maxLength - 1, byteCount);

	ReadArrayU08(pBuffer, copyLength);

	// Zero out any remaining space in the buffer (since the string pulled
	// from the byte stream is not necessarily zero-terminated).
	for (U32 i = copyLength; i < maxLength; ++i) {
		pBuffer[i] = '\0';
	}

	// Skip any chars that could not be read in.
	if (byteCount > copyLength) {
		UtfFormat fmt;
		fmt.AddInt(byteCount);
		fmt.AddInt(maxLength);
		LogErrorMessage("ReadString16() field is %1; bytes, buffer is only %2; wchars", fmt);
		SkipData(byteCount - copyLength);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	AllocString8()
//
Utf08_t* QzByteStream::AllocString8(void)
{
	U32 length = ReadU08();

	if (0 == length) {
		return NULL;
	}

	Utf08_t *pNew = new Utf08_t[length + 1];

	ReadArrayU08(pNew, length);

	pNew[length] = '\0';

	return pNew;
}


/////////////////////////////////////////////////////////////////////////////
//
//	AllocString16()
//
Utf08_t* QzByteStream::AllocString16(void)
{
	U32 length = ReadU16();

	if (0 == length) {
		return NULL;
	}

	Utf08_t *pNew = new Utf08_t[length + 1];

	ReadArrayU08(pNew, length);

	pNew[length] = '\0';

	return pNew;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ChunkCleanup()
//
//	Returns true if there where no obvious problems extracting data from the
//	byte stream.
//
bool QzByteStream::ChunkCleanup(void)
{
	if (m_Overflow) {
		LogErrorMessage("byte stream detected overflow");
	}

	// There is no padding in byte streams.  When we reach the end of the
	// stream, there should be no left over bytes.
	if (0 != m_ByteCount) {
		UtfFormat fmt;
		fmt.AddInt(m_ByteCount);
		LogErrorMessage("byte stream had %1; leftover bytes", fmt);
	}

	return ((false == m_Overflow) && (0 == m_ByteCount));
}


/////////////////////////////////////////////////////////////////////////////
//
//	ComputeCRC()
//
U32 QzByteStream::ComputeCRC(U32 byteCount)
{
	if (byteCount > m_ByteCount) {
		LogErrorMessage("ByteStream::ComputeCRC() overruns end of buffer");
		return 0;
	}

	return QzComputeCrc32(m_pData, byteCount);
}




