/////////////////////////////////////////////////////////////////////////////
//
//	File: QzArray.h
//
//	$Header: /TS/TsGui/QzArray.h  16  2009/9/8 11:59:50a  Lee $
//
//
//	This header defines a basic array template class free from the perils of
//	STL.  This version assumes that it is only dealing with simple data types,
//	such as plain old structs.  So whatever things it contains must not hold
//	references to dynamically allocated memory, otherwise memory corruption
//	will ensue.
//
/////////////////////////////////////////////////////////////////////////////


#pragma once


template <class T> class QzArray
{
private:
	T*  m_pList;
	U32 m_UseCount;
	U32 m_MaxCount;

public:

	/////////////////////////////////////////////////////////////////////////
	//
	//	constructor
	//
	QzArray(void)
		:	m_pList(NULL),
			m_UseCount(0),
			m_MaxCount(0)
	{
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	destructor
	//
	~QzArray(void)
	{
		SafeDeleteArray(m_pList);
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	DeepCopy()
	//
	//	Called when relocating an object to a different location in memory.
	//	It needs to move memory-management responsibility from the source to
	//	here, which means the source's memory pointers need to be NULLed out.
	//
	void DeepCopy(QzArray<T> &source)
	{
		// Free any existing buffer.
		SafeDeleteArray(m_pList);

		// Move source buffer to this object.
		m_pList    = source.m_pList;
		m_UseCount = source.m_UseCount;
		m_MaxCount = source.m_MaxCount;

		// Null out the source's pointer so it's destructor can't free the
		// buffer out from under us.
		source.m_pList    = NULL;
		source.m_UseCount = 0;
		source.m_MaxCount = 0;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	Count()
	//
	//	Returns current count of objects in use.
	//
	U32 Count(void) const
	{
		return m_UseCount;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	MaxCount()
	//
	//	What's the maximum number of elements that could be stored in this
	//	array.  This is misleading, since it ignores the fact that the array
	//	can be reallocated if more space is required.
	//
	U32 MaxCount(void) const
	{
		return m_MaxCount;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	Resize()
	//
	//	Reserves memory for the list, allocating a minimum of the given number
	//	of elements.  If the requested size is smaller than currently in use,
	//	nothing happens.
	//
	void Resize(U32 maximum)
	{
		if (maximum > m_MaxCount) {
			// Keep allocations to multiples of 4.  Start with max(4,...)
			// in case m_MaxCount is zero, the while-loop won't go infinite.

			U32 newMax = Max(4, m_MaxCount);

			while (maximum > newMax) {
				newMax += (newMax / 2);

				// Round to a multiple of 4.
				newMax = (newMax + 3) & 0xFFFFFFFC;
			}

			// Round to a multiple of 4.
			newMax = (newMax + 3) & 0xFFFFFFFC;

			T *pNew = new T[newMax];

			if (0 != m_UseCount) {
				memcpy(pNew, m_pList, sizeof(T) * m_UseCount);
			}

			delete [] m_pList;

			m_pList = pNew;

			m_MaxCount = newMax;
		}
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	SetCount()
	//
	//	Explicitly forces the list to be set to a specific size.  If it is
	//	necessary, memory will be reallocated to support this.
	//
	void SetCount(U32 size)
	{
		Resize(size);

		m_UseCount = size;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	New()
	//
	//	Allocates a new object for use.
	//
	//	WARNING: Since this operation may result in reallocation of array,
	//	any call to New() will potentially invalidate any pointers currently
	//	referencing objects in the list.
	//
	void Append(const T &data)
	{
		// Special case to handle when the list was created without any
		// allocated storage.
		if (0 == m_MaxCount) {
			Resize(4);
		}

		// If the list is not large enough, it needs to grow in size.
		// Default to growing the list 50% each time.  Apparently that is
		// better than doubling, or so claim the swags in the publications.
		if (m_UseCount >= m_MaxCount) {
			Resize(m_MaxCount + (m_MaxCount/2));
		}

		m_pList[m_UseCount++] = data;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	IndexOf()
	//
	//	Returns the index at which the given value is stored, or -1 if the
	//	item is not in the list.
	//
	S32 IndexOf(const T &data)
	{
		for (U32 i = 0; i < m_UseCount; ++i) {
			if (data == m_pList[i]) {
				return i;
			}
		}

		return -1;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	InsertAt()
	//
	//	Inserts data into the middle of the list, shifting all data upwards
	//	in memory one slot to make room for the new item.
	//
	void InsertAt(U32 position, const T &data)
	{
		QzAssert(position <= m_UseCount);

		// Make certain there is enough space in the array.
		// Default to growing the list 50% each time.  Apparently that is
		// better than doubling, or so claim the swags in the publications.
		if (m_UseCount >= m_MaxCount) {
			Resize(m_MaxCount + (m_MaxCount/2));
		}

		++m_UseCount;

		// Shift everything up one slot to make room.
		for (U32 i = m_UseCount - 1; i > position; --i) {
			m_pList[i] = m_pList[i-1];
		}

		m_pList[position] = data;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	DeleteEntry()
	//
	//	Deletes a specific indexed entry from the middle of the list.  All
	//	remaining entries will be shifted down one to fill in the hole.
	//
	void DeleteEntry(U32 position)
	{
		QzAssert(m_UseCount > 0);
		QzAssert(position < m_UseCount);

		for (U32 i = position + 1; i < m_UseCount; ++i) {
			m_pList[i-1] = m_pList[i];
		}

		--m_UseCount;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	DeleteBySwap()
	//
	//	Deletes a specific indexed entry from the middle of the list.  The
	//	last entry in the list will be swapped in to fill the hole, making
	//	for a faster delete, but the ordering is not maintained.
	//
	void DeleteBySwap(U32 position)
	{
		QzAssert(m_UseCount > 0);
		QzAssert(position < m_UseCount);

		// Only do this if we're not deleting the last element.
		if ((position + 1) < m_UseCount) {
			// Swap last element into the deleted slot.
			m_pList[position] = m_pList[m_UseCount - 1];
		}

		--m_UseCount;
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	DeleteByValue()
	//
	//	Removes the first element from the list that has the given value.
	//
	void DeleteByValue(const T &data)
	{
		for (U32 i = 0; i < m_UseCount; ++i) {
			if (data == m_pList[i]) {
				DeleteEntry(i);
				return;
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	Copy()
	//
	//	Copies the elements of the source array into this one.
	//
	//	Note that this is dangerous if the elements contain pointers to
	//	allocated memory, which can result in multiple attempts to free the
	//	same memory.  As such, Copy() should only be used when the elements
	//	do not contain pointers that need to be deleted.
	//
	void Copy(const QzArray<T> &source)
	{
		SetCount(source.Count());

		for (U32 i = 0; i < source.Count(); ++i) {
			m_pList[i] = source.m_pList[i];
		}
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	ReverseEntries()
	//
	//	Reverses the ordering of all elements in the array.
	//
	void ReverseEntries(void)
	{
		if (m_UseCount > 1) {
			for (U32 i = 0, j = m_UseCount - 1; i < j; ++i, --j) {
				Swap(m_pList[i], m_pList[j]);
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	Get()
	//
	//	Returns a pointer to the indexed element in the list, or a NULL
	//	pointer if the index is not valid.
	//
	T& Get(U32 index)
	{
		if (index < m_UseCount) {
			return m_pList[index];
		}

		QzAssert(index < m_UseCount);

		return m_pList[0];
	}


	/////////////////////////////////////////////////////////////////////////
	//
	//	Address()
	//
	//	Returns the address of the underlying buffer.  Should be used with
	//	caution, of course, since it is up to the caller to perform any range
	//	checking before accessing the data.  Plus, adding data to the array
	//	can induce memory reallocation, which would invalidate any cached
	//	addresses.
	//
	T* Address(void)
	{
		return m_pList;
	}
};


